/*
 * Copyright 2005-2009 Freescale Semiconductor, Inc. All rights reserved.
 */

/*
 * The code contained herein is licensed under the GNU General Public
 * License. You may obtain a copy of the GNU General Public License
 * Version 2 or later at the following locations:
 *
 * http://www.opensource.org/licenses/gpl-license.html
 * http://www.gnu.org/copyleft/gpl.html
 */

/*!
 * @file sym.c
 * @brief Test code for Symmetric Cipher support in FSL SHW API
 *
 * This file contains vectors and code to test fsl_shw_symmetric_encrypt(),
 * fsl_shw_symmetric_decrypt(), and the functions associated with the Secret
 * Key and Symmetric Context Objects.
 *
 * @ifnot STANDALONE
 * @ingroup MXCSAHARA2_TEST
 * @endif
 */

#include "api_tests.h"


enum test_type {
	DPD_AES_CTR_ENC,
	DPD_AES_CBC_ENC,
	DPD_AES_ECB_ENC,
	DPD_AES_CTR_DEC,
	DPD_AES_CBC_DEC,
	DPD_AES_ECB_DEC,
	DPD_DES_CBC_ENC,
	DPD_DES_ECB_ENC,
	DPD_DES_CBC_DEC,
	DPD_DES_ECB_DEC,
	DPD_DES_KEY_PARITY,
	DPD_3DES_CBC_ENC,
	DPD_3DES_ECB_ENC,
	DPD_3DES_CBC_DEC,
	DPD_3DES_ECB_DEC,
	DPD_3DES_KEY_PARITY,
	DPD_ARC4_ENC_KEY,
	DPD_ARC4_ENC_SBOX
};

enum test_result {
	PASSED,
	FAILED,
	SKIPPED,
};

typedef struct {
	unsigned long opId;
	unsigned long KeyLen;
	unsigned char *pKey;
	unsigned char *pIV_CTR;	/* IV for CBC, CTR value for CTR, S-Box+ptrs
				   for ARC4 */
	fsl_shw_ctr_mod_t modulus;	/* for CTR mode only */
	unsigned long TextLen;
	unsigned char *pInputText;
	fsl_shw_return_t result;	/* expected return code */
	unsigned char *pOutputText;	/* expected value */
	unsigned char *pOutputIV_CTR;	/* IV for CBC, CTR value for CTR, S-Box+
					   for ARC4 */
	char testDesc[50];
} SYMTESTTYPE;

static unsigned char symtestkey1[] = {
	0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6,
	0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C
};
static unsigned char symtestctr1[] = {
	0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
	0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF
};
static unsigned char symtestinput1[] = {
	0x6B, 0xC1, 0xBE, 0xE2, 0x2E, 0x40, 0x9F, 0x96,
	0xE9, 0x3D, 0x7E, 0x11, 0x73, 0x93, 0x17, 0x2A,
	0xAE, 0x2D, 0x8A, 0x57, 0x1E, 0x03, 0xAC, 0x9C,
	0x9E, 0xB7, 0x6F, 0xAC, 0x45, 0xAF, 0x8E, 0x51,
	0x30, 0xC8, 0x1C, 0x46, 0xA3, 0x5C, 0xE4, 0x11,
	0xE5, 0xFB, 0xC1, 0x19, 0x1A, 0x0A, 0x52, 0xEF
};
static unsigned char symtestoutput1[] = {
	0x87, 0x4D, 0x61, 0x91, 0xB6, 0x20, 0xE3, 0x26,
	0x1B, 0xEF, 0x68, 0x64, 0x99, 0x0D, 0xB6, 0xCE,
	0x98, 0x06, 0xF6, 0x6B, 0x79, 0x70, 0xFD, 0xFF,
	0x86, 0x17, 0x18, 0x7B, 0xB9, 0xFF, 0xFD, 0xFF,
	0x5A, 0xE4, 0xDF, 0x3E, 0xDB, 0xD5, 0xD3, 0x5E,
	0x5B, 0x4F, 0x09, 0x02, 0x0D, 0xB0, 0x3E, 0xAB
};

static unsigned char symtestkey2[] = {
	0x17, 0x10, 0xE3, 0x0D, 0x2C, 0xD9, 0xFB, 0x66,
	0xD2, 0x61, 0x59, 0x22, 0x48, 0x0D, 0xEA, 0xBC
};
static unsigned char symtestinput2[] = {
	0xF3, 0x94, 0x62, 0x33, 0x8D, 0xB0, 0x87, 0xB5,
	0x1E, 0x92, 0xA9, 0x49, 0x37, 0x38, 0x99, 0x3B,
	0x91, 0x76, 0x54, 0x86, 0xB3, 0xA2, 0x51, 0x1C,
	0x29, 0x47, 0x63, 0x88, 0xB7, 0x64, 0x50, 0x2D,
	0x5C, 0x33, 0xB0, 0xBF, 0x76, 0xD4, 0x32, 0x57,
	0xE1, 0xFD, 0xCE, 0xA9, 0x79, 0x03, 0x1B, 0xC4,
	0xB2, 0x42, 0x3E, 0xF0, 0xD0, 0x6E, 0xA4, 0xF1,
	0x6A, 0x9B, 0x11, 0x6D, 0xA7, 0x9F, 0x23, 0x99,
	0x0F, 0xAB, 0xDD, 0x03, 0x35, 0x9D, 0x57, 0x87,
	0x2F, 0x03, 0x1E, 0x5F, 0x13, 0x42, 0x72, 0x46,
	0x6A, 0xE5, 0xE8, 0x9C, 0xEC, 0x77, 0x88, 0xA9,
	0x4E, 0xF5, 0x0E, 0xB9, 0x93, 0x47, 0x54, 0x47,
	0x9D, 0xB6, 0x91, 0x00, 0x72, 0xD9, 0x69, 0xBB,
	0xF0, 0xF5, 0x85, 0x3E, 0x62, 0x44, 0xB7, 0xE0,
	0xBB, 0x14, 0x45, 0xEA, 0xDA, 0x49, 0x7B, 0xD5,
	0xAD, 0x27, 0x12, 0x1E, 0x83, 0xE5, 0x88, 0xF5,
	0x76, 0x04, 0x0B, 0x76, 0x2D, 0xD1, 0xF2, 0xA1,
	0xE7, 0x28, 0x8A, 0xD2, 0x1D, 0xCE, 0x1A, 0xEF,
	0x7C, 0x7B, 0xE1, 0xF6, 0xC6, 0xE6, 0x15, 0x3D,
	0x34, 0xF9, 0x70, 0x03, 0xDA, 0x7B, 0x7C, 0x9A,
	0xDB, 0x3D, 0x22, 0xDB, 0xBB, 0x44, 0x97, 0x1C,
	0xAF, 0xDD, 0x4B, 0x60, 0x4F, 0x1F, 0xE3, 0x07,
	0x5E, 0xBE, 0xDF, 0x91, 0x33, 0xCB, 0x03, 0xE1,
	0x69, 0x32, 0x11, 0x88, 0x51, 0x82, 0x05, 0x66,
	0xEB, 0x02, 0x47, 0x5E, 0xCC, 0x68, 0x11, 0x49,
	0xBA, 0x56, 0x7D, 0xE3, 0x5D, 0xE8, 0x75, 0xEF,
	0xE8, 0x7A, 0xFC, 0x44, 0xFB, 0xEB, 0x0D, 0xFC,
	0xAA, 0x88, 0x77, 0x85, 0xF7, 0xEB, 0x0D, 0xBE,
	0x98, 0xEA, 0x81, 0xE2, 0x69, 0xEE, 0x32, 0x7D,
	0x51, 0xC7, 0x6E, 0x0F, 0x07, 0x5C, 0x46, 0xAE,
	0x7C, 0x44, 0x8C, 0x53, 0x54, 0xB2, 0x10, 0xFE,
	0x30, 0xB0, 0xBC, 0x8B, 0x38, 0x22, 0x9B, 0x42,
	0xB0, 0x88, 0x70, 0x09, 0xEF, 0xFE, 0xE5, 0x47,
	0x96, 0x60, 0x04, 0x50, 0x5F, 0x1D, 0xE8, 0x7E,
	0x51, 0xA8, 0x79, 0xB5, 0xC5, 0x03, 0x05, 0x90,
	0x05, 0x54, 0x91, 0xE0, 0xD3, 0x05, 0xCC, 0xC9,
	0xD6, 0x63, 0x4C, 0x26, 0x17, 0x37, 0x2F, 0x69,
	0x85, 0x47, 0xBB, 0xC9, 0xD3, 0x49, 0x05, 0xD0,
	0x76, 0x2A, 0x47, 0x20, 0x36, 0x39, 0xF8, 0x93,
	0x12, 0x16, 0x41, 0x85, 0xE2, 0xEE, 0xD3, 0x63,
	0x86, 0xFB, 0xE3, 0x47, 0xED, 0xAF, 0x28, 0xE2,
	0xEF, 0x9A, 0xAB, 0x58, 0x2A, 0x75, 0x5B, 0x52,
	0xD6, 0x47, 0x12, 0xF9, 0xDE, 0x28, 0x14, 0x20,
	0x14, 0x90, 0xAC, 0x33, 0xD7, 0xB1, 0x00, 0x55,
	0x15, 0xC9, 0x9F, 0x33, 0xDC, 0xF6, 0x04, 0xE6,
	0x7F, 0x70, 0x80, 0x90, 0x7F, 0xAF, 0xC4, 0xE4,
	0x30, 0x72, 0x8E, 0x67, 0x53, 0x1A, 0x93, 0x86,
	0xA1, 0x55, 0x4D, 0x58, 0x7B, 0x95, 0xB3, 0x1E,
	0xAE, 0x3E, 0x7E, 0x67, 0xA1, 0x16, 0x0D, 0xE2,
	0xB6, 0xD7, 0x80, 0xBC, 0x49, 0x7D, 0x32, 0xA3,
	0x1B, 0x1A, 0x07, 0x40, 0x7E, 0xD4, 0xCD, 0x52,
	0x28, 0xF0, 0x35, 0x1D, 0xED, 0x5C, 0x6C, 0x76,
	0x57, 0xC1, 0x19, 0x1A, 0x55, 0x8A, 0xA5, 0x7C,
	0xEE, 0xD6, 0x8A, 0xE1, 0x54, 0xDC, 0xAB, 0xE0,
	0x08, 0xA1, 0x5F, 0x18, 0xA4, 0x92, 0x33, 0x41,
	0xEF, 0xE3, 0x0D, 0x61, 0xA9, 0x41, 0xBE, 0x4C,
	0xEB, 0xB4, 0x9E, 0x3A, 0x64, 0x4D, 0x49, 0x90,
	0x5C, 0x6C, 0x13, 0xBB, 0xC5, 0x42, 0x52, 0x29,
	0x3F, 0x66, 0x14, 0x39, 0x60, 0x06, 0x4A, 0x4E,
	0x15, 0xA8, 0x18, 0xBC, 0x7F, 0xF2, 0x57, 0xC6,
	0x1F, 0x6F, 0xD6, 0x6B, 0x98, 0xCD, 0x89, 0x33,
	0x07, 0x8E, 0x27, 0xBD, 0x1A, 0x9B, 0x5F, 0x3D,
	0xE1, 0xBA, 0x38, 0xA3, 0xA4, 0x5C, 0xAF, 0xA9,
	0x8E, 0xB2, 0x30, 0x80, 0x99, 0x9C, 0xFB, 0x45,
	0x7D, 0x42, 0x22, 0x0D, 0x10, 0xF1, 0x13, 0xB1,
	0xD2, 0x2B, 0x6D, 0x14, 0x28, 0x4E, 0x22, 0x1A,
	0xE5, 0xEC, 0x77, 0x12, 0xBC, 0x37, 0x1C, 0xBC,
	0x2A, 0x6C, 0xC4, 0xB3, 0x76, 0xDE, 0x87, 0x5D,
	0x6C, 0x75, 0x76, 0x35, 0x41, 0x1A, 0xA6, 0x91,
	0x78, 0x29, 0x24, 0xA3, 0x1B, 0x37, 0x04, 0xF1,
	0x1E, 0x41, 0x16, 0xF6, 0x4B, 0xB2, 0x5D, 0x28,
	0x8E, 0x37, 0xE1, 0x16, 0xF1, 0xD3, 0xEF, 0xDE,
	0x27, 0x49, 0x65, 0xB1, 0xFC, 0x1F, 0x1F, 0x8E,
	0x8B, 0x67, 0x20, 0x07, 0x7A, 0xAC, 0x85, 0x2F,
	0x30, 0xF3, 0xEE, 0x7C, 0x4C, 0x63, 0x5A, 0xC4,
	0x3A, 0x6D, 0x28, 0x20, 0x3B, 0x0D, 0x41, 0xD0,
	0xC1, 0xF6, 0x15, 0x08, 0x68, 0x4D, 0x70, 0x9F,
	0x75, 0xBB, 0xCE, 0x93, 0x22, 0x7B, 0x40, 0x75,
	0x9E, 0x39, 0x75, 0x83, 0x13, 0x53, 0x12, 0xA3,
	0x82, 0x63, 0xD0, 0x01, 0xDD, 0x93, 0xA0, 0x72,
	0x28, 0xAE, 0x46, 0x76, 0x05, 0x6F, 0xA4, 0xEE,
	0x77, 0xF4, 0x36, 0x53, 0x46, 0xE8, 0xE2, 0x9F,
	0xBF, 0x3D, 0xB6, 0xA6, 0x4B, 0x06, 0x9A, 0x0E,
	0x96, 0x61, 0xAE, 0xA1, 0xB8, 0xE7, 0x4A, 0x39,
	0x20, 0x98, 0x50, 0xF2, 0xA8, 0xC2, 0xDD, 0xE3,
	0xAD, 0xD9, 0xF3, 0x0C, 0x76, 0xB2, 0x39, 0xBF,
	0xC4, 0x25, 0x55, 0x38, 0xF5, 0x77, 0x23, 0x84,
	0x7B, 0xAB, 0x2B, 0xA0, 0x06, 0x05, 0x93, 0xD5,
	0x44, 0xD4, 0x23, 0x2B, 0x82, 0xFE, 0x57, 0x1A,
	0x0A, 0x27, 0x3E, 0x38, 0x95, 0x0F, 0x21, 0x22,
	0xB4, 0x0D, 0x8D, 0x41, 0x70, 0x1B, 0xF2, 0xBF,
	0x10, 0x7B, 0x46, 0x57, 0x57, 0x5B, 0xE8, 0x2E,
	0x09, 0x7D, 0x45, 0x85, 0x19, 0x54, 0x63, 0x66,
	0x54, 0x95, 0xE2, 0x0D, 0xE2, 0xAA, 0x93, 0x49,
	0x71, 0x09, 0x32, 0x83, 0x6C, 0xDB, 0x66, 0xB0,
	0x0B, 0x03, 0x9C, 0xD4, 0x94, 0xD4, 0xCD, 0x62,
	0xBB, 0xA0, 0xD7, 0x20, 0x4D, 0x69, 0x6D, 0xD6,
	0x32, 0xD1, 0x46, 0x75, 0xF0, 0xAA, 0xA2, 0xEE,
	0xB1, 0x26, 0xB0, 0x7A, 0xF5, 0x1E, 0xF6, 0x81,
	0xF8, 0x6E, 0x5D, 0xE1, 0x00, 0xD2, 0xE0, 0xC7,
	0x7E, 0x48, 0x91, 0xCF, 0x55, 0x5E, 0xF6, 0xAE,
	0x1B, 0x83, 0x67, 0x16, 0xAF, 0xAD, 0x76, 0x07,
	0x07, 0x68, 0x0B, 0x56, 0x70, 0xBB, 0x31, 0x91,
	0x41, 0xDD, 0x53, 0xFC, 0x33, 0x2F, 0xD5, 0xED,
	0x4F, 0x74, 0xC2, 0x23, 0xC0, 0xCA, 0x9B, 0x6D,
	0x62, 0x48, 0xDA, 0x4B, 0x63, 0xC6, 0x50, 0xBD,
	0xD8, 0xCB, 0xDC, 0x03, 0x96, 0x0A, 0xBD, 0x7A,
	0x22, 0x6D, 0xE0, 0x5F, 0x1E, 0x3A, 0x79, 0x9B,
	0x02, 0x1E, 0x53, 0x60, 0x76, 0xBD, 0x0E, 0xC4,
	0x32, 0xB7, 0xCF, 0x27, 0xA8, 0x86, 0x87, 0x73,
	0x67, 0x46, 0x5B, 0x20, 0x7B, 0xD2, 0x59, 0x0D,
	0xB1, 0x2F, 0xFD, 0xF8, 0x0B, 0xBE, 0xAF, 0xD0,
	0x42, 0x34, 0xC1, 0x84, 0xB5, 0xBD, 0x19, 0xA4,
	0x8C, 0x5E, 0x0B, 0x77, 0x74, 0xEF, 0x8C, 0xC4,
	0xCA, 0xC3, 0x50, 0x09, 0x8A, 0x5A, 0xDB, 0x52,
	0xDD, 0x2D, 0x3D, 0x71, 0x99, 0xFC, 0x77, 0xC2,
	0x94, 0xA0, 0x27, 0x46, 0x16, 0xCB, 0x9E, 0x30,
	0x4C, 0xC3, 0xE8, 0xBF, 0x15, 0x81, 0xE9, 0x83,
	0xF2, 0x27, 0x21, 0xCF, 0x87, 0x5B, 0x33, 0x8B,
	0x6C, 0x6A, 0xC8, 0x28, 0xC4, 0xAE, 0xE2, 0xE1,
	0x55, 0x43, 0x32, 0x14, 0x85, 0x5D, 0x9A, 0xC4,
	0x22, 0x66, 0x61, 0x3B, 0x2F, 0x30, 0x3F, 0xBC,
	0xA8, 0x4E, 0xC8, 0x3D, 0x8B, 0x0B, 0x67, 0x31,
	0xFC, 0xE0, 0x62, 0x35, 0xDC, 0x06, 0x20, 0xD4,
	0xB7, 0xF4, 0x2B, 0x12, 0x4C, 0x64, 0x1E, 0xF6,
	0x99, 0xBC, 0x02, 0xDD, 0xBF, 0x69, 0x45, 0xB0,
	0xF3, 0x94, 0x62, 0x33, 0x8D, 0xB0, 0x87, 0xB5,
	0x1E, 0x92, 0xA9, 0x49, 0x37, 0x38, 0x99, 0x3B
};

static unsigned char symtestoutput2[] = {
	0xAE, 0x2D, 0x35, 0xC9, 0xB9, 0x03, 0x0E, 0xD0,
	0x96, 0xF5, 0xE7, 0xEE, 0x42, 0xA8, 0x11, 0x84,
	0x2E, 0x1B, 0x95, 0xA1, 0x2B, 0xAF, 0x3C, 0xBB,
	0xB1, 0x1B, 0x31, 0xFA, 0x19, 0x68, 0xAA, 0x18,
	0x32, 0x94, 0x4A, 0xAB, 0x08, 0x3F, 0xC4, 0x8E,
	0xB1, 0xB6, 0x94, 0x06, 0x5C, 0x9A, 0x60, 0x4B,
	0xDA, 0x84, 0xFD, 0xDD, 0x28, 0xB8, 0x20, 0x46,
	0x30, 0xC8, 0xD2, 0x6C, 0xA3, 0x88, 0x86, 0x29,
	0xE7, 0x89, 0xC1, 0x05, 0x19, 0x32, 0x81, 0xEC,
	0xBA, 0x73, 0x7B, 0x82, 0xDD, 0x2F, 0x4C, 0xC0,
	0xAD, 0xD1, 0x0B, 0x2F, 0x1C, 0x88, 0xBD, 0x74,
	0x2E, 0xD3, 0x5B, 0x82, 0xD6, 0x32, 0x43, 0x9A,
	0x59, 0x56, 0x9C, 0x0C, 0x5D, 0x8F, 0x22, 0x74,
	0x62, 0x17, 0xA2, 0x33, 0x37, 0x09, 0x44, 0x12,
	0xF9, 0x40, 0xA2, 0x47, 0x06, 0x9E, 0xF7, 0xFD,
	0xBA, 0x5F, 0x68, 0x02, 0x91, 0x30, 0x84, 0x76,
	0x1F, 0x4D, 0xDA, 0x41, 0xC7, 0x84, 0x8A, 0x04,
	0xE5, 0x87, 0xDC, 0x87, 0xEB, 0xD9, 0xAE, 0x4C,
	0xDD, 0xDE, 0x84, 0xB8, 0x95, 0x42, 0x6B, 0xB8,
	0x03, 0x90, 0x06, 0x72, 0xA6, 0x67, 0xD6, 0xD6,
	0xD8, 0xF7, 0x52, 0x9E, 0x2F, 0x39, 0x1B, 0xDE,
	0x48, 0xD0, 0xD5, 0x1B, 0xE4, 0x75, 0xF8, 0x8E,
	0xFE, 0xBA, 0x6B, 0xFC, 0x35, 0x24, 0x07, 0x42,
	0x31, 0x07, 0x1C, 0x7C, 0x1E, 0xEF, 0x65, 0x15,
	0xF7, 0x87, 0x41, 0x1B, 0x7F, 0x02, 0x28, 0x32,
	0xA5, 0x52, 0xBD, 0xA3, 0xC4, 0xB6, 0x25, 0xFB,
	0x56, 0xB7, 0xE5, 0x96, 0xB4, 0x95, 0xBB, 0xCE,
	0x30, 0x59, 0xC9, 0x23, 0x4A, 0x2C, 0xFC, 0x1F,
	0x3E, 0xB8, 0x30, 0x6A, 0x0A, 0x37, 0x0C, 0x8B,
	0x2F, 0xC9, 0x9C, 0x6A, 0x92, 0x00, 0x22, 0x03,
	0xB3, 0xDF, 0xE2, 0xAE, 0x0D, 0xBA, 0x96, 0xF7,
	0x85, 0xC5, 0x82, 0x38, 0x07, 0x80, 0x64, 0xF4,
	0x86, 0xBC, 0xBD, 0xD8, 0x6E, 0xEC, 0x59, 0x07,
	0x5F, 0x64, 0xFC, 0xA6, 0x43, 0x1F, 0xC8, 0x7C,
	0x6A, 0xFF, 0x85, 0x8A, 0x17, 0x7E, 0x52, 0x22,
	0x76, 0x35, 0xB8, 0x6B, 0x13, 0xBC, 0x2F, 0x64,
	0x19, 0xC8, 0x22, 0x4C, 0x98, 0x0C, 0x06, 0xFB,
	0xB7, 0x8E, 0x05, 0x93, 0x42, 0xC2, 0x2C, 0x1D,
	0x00, 0x2B, 0xED, 0xF4, 0x0E, 0xF4, 0x43, 0x2C,
	0xDF, 0x5A, 0xCC, 0x12, 0x8C, 0xD1, 0xA6, 0xA9,
	0x07, 0x1F, 0xAD, 0xF0, 0xD4, 0x2C, 0xEB, 0xD7,
	0xD3, 0x3E, 0x2F, 0x8D, 0xAB, 0xBB, 0xCE, 0x58,
	0xCE, 0xC0, 0xDD, 0x64, 0xAB, 0x3F, 0xA4, 0x68,
	0x86, 0xDE, 0x10, 0x54, 0x4C, 0xC5, 0x80, 0xDA,
	0xDD, 0x91, 0xA6, 0x3E, 0xAF, 0x1A, 0x74, 0x6E,
	0x3C, 0x3C, 0x64, 0x54, 0xEA, 0x83, 0x02, 0x9B,
	0x82, 0x35, 0xD3, 0x3E, 0x17, 0x50, 0x5F, 0xB5,
	0x5C, 0x0F, 0xA3, 0xDB, 0xAD, 0x3F, 0x81, 0xDC,
	0x4E, 0xBF, 0x0C, 0xD0, 0x2D, 0x21, 0x99, 0x94,
	0x48, 0x8A, 0x3E, 0xD6, 0x6F, 0x08, 0x9B, 0x9A,
	0x3A, 0x60, 0xEF, 0x91, 0x5D, 0x86, 0xF1, 0xDE,
	0xD0, 0x98, 0x18, 0xAC, 0xB2, 0xBC, 0x8A, 0xA6,
	0x6F, 0x96, 0xF1, 0x04, 0x26, 0x8B, 0x9C, 0x67,
	0x96, 0x15, 0x35, 0x35, 0x88, 0x55, 0x84, 0xAF,
	0xBB, 0xB2, 0x2C, 0x4C, 0x28, 0xB9, 0xEE, 0x05,
	0xCB, 0x65, 0xAF, 0x59, 0x2E, 0xD1, 0x1C, 0x0B,
	0x28, 0xA3, 0x90, 0x6F, 0x78, 0x9D, 0xB4, 0x0D,
	0x43, 0x57, 0x83, 0x4E, 0x39, 0x87, 0x66, 0xB1,
	0x3F, 0x5B, 0xF4, 0x9D, 0xE7, 0x7A, 0xBA, 0x6D,
	0x54, 0x57, 0x2F, 0x0E, 0xA6, 0xAA, 0x3B, 0xB7,
	0xC3, 0x96, 0xB9, 0x47, 0x72, 0xCC, 0xF5, 0xBF,
	0x19, 0xD1, 0x18, 0xE0, 0x59, 0x95, 0x9C, 0x39,
	0x18, 0x41, 0x37, 0xBD, 0x37, 0x95, 0x56, 0xAF,
	0xBA, 0x9C, 0x9D, 0xF7, 0x03, 0x06, 0x4D, 0x76,
	0xB3, 0x87, 0x12, 0x69, 0x32, 0x95, 0x61, 0x89,
	0x0A, 0x9D, 0xEC, 0xD2, 0x2B, 0xF2, 0x47, 0xCB,
	0x37, 0x26, 0x3E, 0x51, 0xDA, 0x56, 0x5C, 0x39,
	0xA3, 0x27, 0xF0, 0xEE, 0xEE, 0x26, 0xF9, 0x9F,
	0x8F, 0xC3, 0xC8, 0x40, 0x92, 0xCE, 0xE2, 0xD8,
	0x76, 0xF3, 0x79, 0xD2, 0x4B, 0xD9, 0x51, 0xCA,
	0xCE, 0x11, 0x45, 0xBC, 0x9B, 0x28, 0x64, 0x05,
	0x14, 0x74, 0x56, 0x22, 0x0E, 0xAF, 0xF9, 0xB9,
	0x5F, 0xCA, 0x95, 0x74, 0xFB, 0x08, 0x45, 0x45,
	0x88, 0xE9, 0x39, 0x55, 0x2A, 0x48, 0x6F, 0xC0,
	0xA5, 0xF3, 0x02, 0x85, 0xA5, 0x0D, 0xF2, 0x98,
	0x52, 0x2A, 0x46, 0x19, 0x15, 0xD1, 0xCA, 0xEC,
	0x03, 0x69, 0x9B, 0xFE, 0x43, 0xC2, 0xCD, 0x1E,
	0xAC, 0xA7, 0x2F, 0xD4, 0xAC, 0x8C, 0x95, 0x88,
	0xCA, 0xE8, 0x87, 0x2E, 0x86, 0xEB, 0x21, 0x90,
	0xC2, 0xA8, 0x4F, 0xF5, 0x76, 0x34, 0xFE, 0xE1,
	0x91, 0xCE, 0x5B, 0xE2, 0x07, 0x25, 0x00, 0x17,
	0x09, 0x9D, 0x1F, 0x4B, 0x03, 0xEF, 0x5B, 0x9C,
	0x3C, 0x74, 0x9F, 0xD1, 0x05, 0xEB, 0x64, 0x96,
	0xD9, 0xA3, 0xC6, 0x26, 0x5A, 0xE5, 0x6C, 0x84,
	0x27, 0x92, 0x55, 0x6C, 0xB2, 0x04, 0x1C, 0xDD,
	0xA1, 0xE0, 0x77, 0xE9, 0xA2, 0x18, 0x86, 0x41,
	0x0F, 0x75, 0x09, 0x3F, 0x04, 0x93, 0xFC, 0x5B,
	0x57, 0xC3, 0x98, 0x04, 0xE8, 0xA0, 0x39, 0x92,
	0xB4, 0xD2, 0x09, 0xAA, 0xD8, 0xE6, 0xD0, 0xA0,
	0xBB, 0x5C, 0x3A, 0x63, 0xAE, 0xBC, 0xBA, 0xCC,
	0xC5, 0xEE, 0xF6, 0x90, 0xC7, 0x82, 0xFA, 0xC6,
	0x5A, 0x33, 0x71, 0x12, 0xE6, 0x1C, 0x57, 0xFA,
	0x38, 0x23, 0xE2, 0x6C, 0x14, 0x4B, 0xBC, 0x0E,
	0x09, 0x96, 0x9E, 0xC4, 0xA5, 0x91, 0x0E, 0xBF,
	0xCA, 0x05, 0xA6, 0xE1, 0x1E, 0xC6, 0xC3, 0x1B,
	0x69, 0x44, 0xCC, 0x8D, 0x7B, 0x1C, 0x63, 0xED,
	0x80, 0x63, 0x44, 0x01, 0xCB, 0xB0, 0x93, 0x4F,
	0x5D, 0xA8, 0x61, 0x5E, 0x73, 0x51, 0xA8, 0x29,
	0x61, 0x41, 0xB6, 0xBC, 0x7E, 0x00, 0x2C, 0xB8,
	0xDF, 0xBF, 0x8A, 0x5C, 0xD9, 0xC8, 0x4B, 0x31,
	0x95, 0x44, 0x86, 0x4C, 0xC1, 0xA6, 0xFF, 0x31,
	0x73, 0x9F, 0x1A, 0xB4, 0x02, 0x9E, 0x37, 0x09,
	0x97, 0x55, 0xD5, 0x39, 0xA1, 0x1A, 0x2E, 0x21,
	0x1F, 0x91, 0xE6, 0x91, 0x1A, 0xD1, 0x79, 0xC7,
	0xF6, 0x57, 0xC7, 0xC1, 0x73, 0xFB, 0xE4, 0x91,
	0x61, 0x38, 0xC0, 0x66, 0xF5, 0x05, 0x0D, 0xCE,
	0x99, 0x82, 0x78, 0x37, 0x63, 0x0D, 0x00, 0x04,
	0x4C, 0xCB, 0xFA, 0x32, 0x0B, 0xD0, 0xDE, 0x7C,
	0x34, 0x58, 0xCC, 0xCD, 0x2D, 0x27, 0xA2, 0x41,
	0x7A, 0x8D, 0x82, 0x3A, 0x0C, 0x0E, 0x0F, 0xE9,
	0x58, 0x90, 0x46, 0x44, 0x7A, 0xA0, 0xE1, 0x6B,
	0x21, 0x33, 0xCB, 0x73, 0xAE, 0x50, 0x0C, 0xBB,
	0xF3, 0x6C, 0x0B, 0xB7, 0xF8, 0x7D, 0xAD, 0x7C,
	0x11, 0xF9, 0x5A, 0x15, 0x1E, 0xB4, 0xBE, 0xFE,
	0x38, 0x65, 0x9B, 0xE6, 0xD1, 0x65, 0x73, 0x9D,
	0x76, 0x6F, 0x03, 0xF2, 0x0C, 0x71, 0x72, 0xC5,
	0xB5, 0x1C, 0x4F, 0xE1, 0xE5, 0x2E, 0xD1, 0x56,
	0xE2, 0xED, 0xAE, 0x40, 0x1E, 0x97, 0x23, 0x2C,
	0x58, 0x96, 0x8B, 0xA0, 0xE2, 0x45, 0xC6, 0x00,
	0xC9, 0xAC, 0x2A, 0xAF, 0x0B, 0x7E, 0xBA, 0x84,
	0xF1, 0xF3, 0x9D, 0x8E, 0x1E, 0xEE, 0xA8, 0xF5,
	0x91, 0x3F, 0xF9, 0x93, 0x70, 0x91, 0xCD, 0x09,
	0x84, 0xB2, 0x11, 0xFB, 0x58, 0x76, 0x70, 0x36,
	0x4F, 0xD9, 0x12, 0x57, 0xD1, 0xAA, 0x1B, 0xAE,
	0xE2, 0xE3, 0xA3, 0x7D, 0x6F, 0xA4, 0xE1, 0x4C,
	0x2D, 0xE9, 0x22, 0xEF, 0x9B, 0x80, 0x95, 0x32,
	0xAE, 0x2D, 0x35, 0xC9, 0xB9, 0x03, 0x0E, 0xD0,
	0x96, 0xF5, 0xE7, 0xEE, 0x42, 0xA8, 0x11, 0x84
};

static unsigned char symtestkey3[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};
static unsigned char symtestinput3[] = {
	0x66, 0xe9, 0x4b, 0xd4, 0xef, 0x8a, 0x2c, 0x3b,
	0x88, 0x4c, 0xfa, 0x59, 0xca, 0x34, 0x2b, 0x2e,
	0xf7, 0x95, 0xbd, 0x4a, 0x52, 0xe2, 0x9e, 0xd7,
	0x13, 0xd3, 0x13, 0xfa, 0x20, 0xe9, 0x8d, 0xbc,
	0x10, 0xa5, 0x82, 0xed, 0x9b, 0xd1, 0x5d, 0xbe,
	0x6a, 0x6e, 0xe4, 0xb1, 0x58, 0xd5, 0x79, 0x0e,
	0x75, 0x4b, 0x44, 0xfa, 0x7b, 0xae, 0x0b, 0xb2,
	0xf4, 0x82, 0xf2, 0xf2, 0x63, 0xa4, 0x24, 0x00
};
static unsigned char symtestoutput3[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x66, 0xe9, 0x4b, 0xd4, 0xef, 0x8a, 0x2c, 0x3b,
	0x88, 0x4c, 0xfa, 0x59, 0xca, 0x34, 0x2b, 0x2e,
	0xf7, 0x95, 0xbd, 0x4a, 0x52, 0xe2, 0x9e, 0xd7,
	0x13, 0xd3, 0x13, 0xfa, 0x20, 0xe9, 0x8d, 0xbc
};

unsigned char symtestkey4[] = {
	0xCC, 0xB4, 0x8F, 0xD3, 0xF9, 0x0B, 0xF2, 0x96
};
unsigned char symtestinput4[] = {
	0xCA, 0xCE, 0x0F, 0x3A, 0x37, 0xF6, 0xDD, 0x9A,
	0xEB, 0x04, 0x30, 0x23, 0x6C, 0xF2, 0x85, 0xC5,
	0xC2, 0x7D, 0x6C, 0xCF, 0x81, 0x08, 0x5D, 0x52,
	0x0A, 0x23, 0x5D, 0x67, 0xCF, 0x82, 0x27, 0x3E,
	0xAC, 0x91, 0xD7, 0xD9, 0x67, 0xD0, 0x4B, 0x7A,
	0x97, 0xE2, 0x62, 0xCE, 0xBA, 0x51, 0xAF, 0x90,
	0x9F, 0xEC, 0x7D, 0x46, 0xB6, 0xFB, 0x7C, 0xD1,
	0x35, 0x00, 0xC8, 0xFD, 0x06, 0xB1, 0xA5, 0x54,
	0xCB, 0x6A, 0x1A, 0x19, 0x6A, 0xA1, 0x67, 0x65,
	0xFD, 0xEF, 0x29, 0x8B, 0x70, 0xA8, 0xE4, 0x34,
	0x25, 0x62, 0xFE, 0x0F, 0xBA, 0x8B, 0xF6, 0xCA,
	0xDC, 0x1E, 0x64, 0x9F, 0x83, 0x1D, 0xDF, 0xDF,
	0x50, 0xE9, 0xCA, 0xA7, 0x41, 0xFF, 0xEC, 0xD3,
	0x6B, 0xD4, 0xAB, 0x10, 0x48, 0x02, 0x93, 0xB9,
	0x29, 0x31, 0x32, 0x62, 0x92, 0xF6, 0x8F, 0x14,
	0xF8, 0x1B, 0x3C, 0x56, 0xB8, 0xD0, 0xE1, 0xF7,
	0x75, 0xCD, 0xB1, 0x74, 0xD3, 0xCB, 0x1D, 0x9A,
	0x07, 0xB3, 0xD8, 0xB6, 0x4C, 0x71, 0x18, 0xAD,
	0xC8, 0x12, 0xEF, 0x03, 0x0F, 0x7F, 0xCE, 0xB1,
	0xFF, 0x86, 0xC3, 0x85, 0x57, 0xBF, 0x6B, 0xB8,
	0x3F, 0xAD, 0xAC, 0x74, 0xF5, 0x9C, 0x51, 0xD5,
	0xB1, 0x13, 0xBE, 0x04, 0x9D, 0x70, 0x43, 0x2D,
	0x09, 0x5F, 0xD7, 0x2D, 0x9D, 0x3D, 0xB5, 0x70,
	0xFA, 0x89, 0x2A, 0x7B, 0x44, 0x7A, 0xB0, 0x6A
};
unsigned char symtestoutput4[] = {
	0x4F, 0x0D, 0x4C, 0xDC, 0x80, 0xC0, 0x5C, 0xED,
	0xA0, 0xF4, 0xF6, 0xC0, 0x9B, 0x3A, 0x2B, 0x9E,
	0x1F, 0xA1, 0x40, 0x38, 0x46, 0x1A, 0xA3, 0x86,
	0x52, 0x40, 0x3A, 0xC9, 0xB9, 0x94, 0x6E, 0xC4,
	0x99, 0xC0, 0x07, 0x18, 0xF2, 0x10, 0x81, 0xBB,
	0xC6, 0x04, 0xE8, 0x1D, 0x6E, 0x72, 0xAB, 0xA3,
	0x14, 0xA2, 0x1B, 0x7E, 0x19, 0x6E, 0xA1, 0xB5,
	0x49, 0xED, 0x8A, 0xDE, 0xCB, 0x2D, 0xC2, 0xED,
	0x01, 0x56, 0xC5, 0x8A, 0xF8, 0x66, 0x3F, 0xDF,
	0x21, 0x2C, 0x09, 0x5A, 0xAA, 0xF2, 0x2B, 0x91,
	0x9C, 0x83, 0xA1, 0xA8, 0x16, 0x01, 0xB9, 0xAE,
	0xCC, 0xEA, 0xA2, 0x1B, 0xF0, 0xB3, 0xBB, 0xDD,
	0x17, 0xA1, 0xDC, 0x74, 0xD4, 0x2C, 0x6E, 0x07,
	0x45, 0x02, 0x53, 0xCB, 0x8A, 0x1A, 0xBA, 0x1B,
	0xB2, 0xE1, 0xF4, 0xD4, 0x9C, 0x1B, 0xCF, 0xE9,
	0xFD, 0xF8, 0x5B, 0x71, 0x5E, 0x54, 0x02, 0x03,
	0xC3, 0x28, 0xF9, 0xEE, 0xA2, 0x35, 0xEE, 0xA5,
	0x00, 0xF3, 0x8A, 0x32, 0x0D, 0x83, 0xDF, 0x7B,
	0x6B, 0x00, 0x6C, 0xD6, 0x42, 0x39, 0x3F, 0xA7,
	0x84, 0x01, 0x03, 0x7D, 0x71, 0x62, 0xFB, 0x75,
	0xB5, 0xC9, 0x31, 0x2B, 0xB3, 0xEB, 0xC8, 0x87,
	0xB8, 0xD4, 0xA9, 0xE4, 0x5D, 0x8E, 0xA6, 0x53,
	0xC3, 0x8A, 0xFB, 0xAB, 0x46, 0x84, 0xFE, 0x5E,
	0x95, 0xC6, 0x92, 0x42, 0x6E, 0xFC, 0x60, 0xA1
};
unsigned char symtestkey5[] = {	/* TEMPORARY patch */
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01
};
unsigned char symtestinput5[] = {
	0x95, 0xF8, 0xA5, 0xE5, 0xDD, 0x31, 0xD9, 0x00,
	0xDD, 0x7F, 0x12, 0x1C, 0xA5, 0x01, 0x56, 0x19,
	0x2E, 0x86, 0x53, 0x10, 0x4F, 0x38, 0x34, 0xEA,
	0x4B, 0xD3, 0x88, 0xFF, 0x6C, 0xD8, 0x1D, 0x4F,
	0x20, 0xB9, 0xE7, 0x67, 0xB2, 0xFB, 0x14, 0x56,
	0x55, 0x57, 0x93, 0x80, 0xD7, 0x71, 0x38, 0xEF,
	0x6C, 0xC5, 0xDE, 0xFA, 0xAF, 0x04, 0x51, 0x2F,
	0x0D, 0x9F, 0x27, 0x9B, 0xA5, 0xD8, 0x72, 0x60
};
unsigned char symtestoutput5[] = {
	0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

unsigned char symtestkey6[] = {
	0x65, 0xA4, 0xD7, 0xD2
};
unsigned char symtestinput6[] = {
	0x46, 0x73, 0xF7, 0x2F, 0x53, 0x9F, 0x20, 0x69,
	0xD0, 0xA7, 0x00, 0xD3, 0x0F, 0x12, 0xA3, 0xB1,
	0xD8, 0x88, 0x17, 0xC3, 0x49, 0xE1, 0xC4, 0x83,
	0x44, 0xD7, 0xD3, 0xF2, 0x4E, 0xA1, 0x22, 0x2A,
	0xEE, 0x67, 0xA7, 0x1F, 0x6A, 0xD5, 0xE9, 0xDD,
	0xF5, 0xB3, 0x47, 0xAD, 0x2A, 0xC6, 0xB7, 0x4F,
	0xE9, 0xEB, 0xEC, 0xE3, 0x74, 0xB3, 0x43, 0x91,
	0xDA, 0x56, 0x87, 0x85, 0x8F, 0x66, 0x00, 0x66,
	0x4B, 0x0E, 0xB9, 0xCD, 0xFB, 0x28, 0x8B, 0xE7,
	0xFA, 0x2A, 0xC4, 0xC8, 0x37, 0x10, 0x0C, 0x95,
	0x99, 0x42, 0x50, 0x13, 0x55, 0xB9, 0x62, 0xBC,
	0x53, 0x49, 0x0E, 0xED
};
unsigned char symtestoutput6[] = {
	0xC9, 0x99, 0xB8, 0x6F, 0x59, 0x52, 0xAD, 0xB5,
	0x22, 0x83, 0xA1, 0x76, 0x48, 0xCD, 0x1E, 0x24,
	0x69, 0xEE, 0x45, 0xBB, 0x12, 0x5F, 0x0B, 0x83,
	0x5A, 0x00, 0x80, 0x19, 0xCF, 0x9B, 0x45, 0xD2,
	0xBF, 0x8A, 0x44, 0x48, 0x68, 0x21, 0x25, 0x45,
	0x9C, 0x2B, 0xD8, 0x1A, 0x4B, 0xE8, 0x25, 0xCB,
	0xF4, 0x9C, 0xF4, 0xFE, 0xE9, 0x9B, 0x72, 0x89,
	0x69, 0x61, 0xB3, 0x0B, 0x15, 0xDC, 0xF4, 0xDC,
	0x2B, 0x6E, 0xDD, 0x2C, 0x43, 0x31, 0xA4, 0xE5,
	0x1A, 0x56, 0x89, 0xA5, 0xC9, 0xCD, 0x83, 0x3E,
	0x62, 0x79, 0xCD, 0xED, 0xE0, 0xA7, 0x57, 0x09,
	0x10, 0x1D, 0x71, 0xEC
};
unsigned char symtestsbox7[] = {
	0x65, 0x38, 0xA2, 0xA8, 0xA9, 0x04, 0x05, 0xB1,
	0xE8, 0xA3, 0xEF, 0x3F, 0x86, 0xC1, 0xC7, 0xCE,
	0xE0, 0xA6, 0x32, 0x58, 0xCC, 0x2C, 0xD0, 0x5D,
	0x9C, 0x2D, 0xE5, 0x1C, 0xEC, 0x17, 0xFA, 0x4E,
	0xB0, 0x27, 0xD5, 0xE1, 0x5C, 0xAE, 0x10, 0x28,
	0x5B, 0x35, 0x41, 0xDE, 0x31, 0xB9, 0xDA, 0xE2,
	0x98, 0x74, 0x99, 0xAC, 0xDF, 0x75, 0x9E, 0xEB,
	0xA7, 0x84, 0x11, 0x5E, 0xCB, 0xE6, 0x02, 0xC0,
	0xBF, 0x45, 0x6F, 0xA5, 0x20, 0x0E, 0xCA, 0xD9,
	0x2F, 0xF7, 0x3C, 0xCF, 0x0D, 0x00, 0x60, 0xAA,
	0xD2, 0xC8, 0x72, 0x26, 0x54, 0x19, 0x94, 0x70,
	0xBB, 0x83, 0x62, 0x9A, 0x42, 0x4D, 0x5F, 0x01,
	0x92, 0xDC, 0xCD, 0xE3, 0xA4, 0x0C, 0xBC, 0x57,
	0x73, 0x07, 0x85, 0x47, 0x8E, 0x16, 0xE4, 0x0B,
	0x7E, 0x96, 0xED, 0x2E, 0x23, 0x8B, 0xD8, 0x8A,
	0x12, 0x7D, 0xD6, 0xF2, 0x8F, 0x1B, 0xF6, 0xFC,
	0x3A, 0x33, 0x4F, 0x77, 0x79, 0x4C, 0x91, 0xBD,
	0x52, 0x36, 0xC9, 0xD1, 0x13, 0x6D, 0xA1, 0x66,
	0x9B, 0xF1, 0x06, 0xB3, 0xF8, 0xFB, 0x48, 0xF0,
	0xC5, 0x46, 0x90, 0x63, 0xB8, 0x1F, 0xB2, 0x8C,
	0xBA, 0xD7, 0x69, 0x34, 0x29, 0xE9, 0xEE, 0xC6,
	0x81, 0xAF, 0x3B, 0x30, 0xE7, 0x6E, 0x71, 0xF9,
	0x0F, 0xFD, 0x1A, 0x39, 0xA0, 0xF3, 0x18, 0x1E,
	0x56, 0x43, 0x55, 0x53, 0x40, 0x03, 0xD3, 0xC4,
	0x9D, 0x88, 0xDD, 0x2B, 0x4B, 0x67, 0x93, 0x59,
	0x1D, 0x7C, 0x7F, 0x51, 0xF4, 0x44, 0x61, 0x21,
	0xDB, 0x68, 0xAB, 0x7A, 0x78, 0x14, 0x3E, 0xFE,
	0xBE, 0xAD, 0x7B, 0x89, 0x5A, 0x37, 0x15, 0xB4,
	0xF5, 0xB5, 0x09, 0xD4, 0x3D, 0x82, 0x76, 0xFF,
	0x6C, 0x25, 0x50, 0xB6, 0x80, 0x9F, 0xC2, 0x97,
	0x49, 0x22, 0xEA, 0x6B, 0xB7, 0x08, 0x95, 0x87,
	0x4A, 0x24, 0x64, 0xC3, 0x0A, 0x8D, 0x2A, 0x6A,
	0x5D, 0xC8, 0x4D, 0x00
};
static unsigned char symtestinput7[] = {
	0x85, 0x55, 0x45, 0x44, 0x40, 0xDA, 0x44, 0xC2,
	0x6D, 0x1F, 0x33, 0xDA, 0xB4, 0xD7, 0x31, 0xF4,
	0x6A, 0x2D, 0x4D, 0x7C, 0x6C, 0x69, 0x56, 0x25,
	0xAE, 0xA7, 0x0D, 0x62, 0xA0, 0x68, 0x9B, 0xC6,
	0x40, 0x56, 0xDB, 0x49, 0xEF, 0x1B, 0x58, 0x1D,
	0xDC, 0x97, 0x88, 0x21, 0x2F, 0x5C, 0x00, 0x00
};

static unsigned char symtestoutput7[] = {
	0xF8, 0x1A, 0x56, 0xE1, 0x4F, 0x80, 0x0C, 0xA4,
	0x90, 0x0F, 0x85, 0x4D, 0xBD, 0x65, 0x8D, 0xD5,
	0x7E, 0x3B, 0xD2, 0xFC, 0x05, 0x25, 0x9E, 0x2A,
	0xD6, 0x17, 0x1C, 0x52, 0xE8, 0xBD, 0xF2, 0x3E,
	0x2F, 0x70, 0x8F, 0x58, 0x43, 0x75, 0x1D, 0xFB,
	0xB4, 0x40, 0xCE, 0x78, 0xED, 0xDD, 0x00, 0x00
};

static unsigned char symtestkey8[] = {
	0x65, 0xA4, 0xD7, 0xD2
};
static unsigned char symtestinput8[] = {
	0x46, 0x73, 0xF7, 0x2F, 0x53, 0x9F, 0x20, 0x69,
	0xD0, 0xA7, 0x00, 0xD3, 0x0F, 0x12, 0xA3, 0xB1,
	0xD8, 0x88, 0x17, 0xC3, 0x49, 0xE1, 0xC4, 0x83,
	0x44, 0xD7, 0xD3, 0xF2, 0x4E, 0xA1, 0x22, 0x2A,
	0xEE, 0x67, 0xA7, 0x1F, 0x6A, 0xD5, 0xE9, 0xDD,
	0xF5, 0xB3, 0x47, 0xAD, 0x2A, 0xC6, 0xB7, 0x4F,
	0xE9, 0xEB, 0xEC, 0xE3, 0x74, 0xB3, 0x43, 0x91,
	0xDA, 0x56, 0x87, 0x85, 0x8F, 0x66, 0x00, 0x66,
	0x4B, 0x0E, 0xB9, 0xCD, 0xFB, 0x28, 0x8B, 0xE7,
	0xFA, 0x2A, 0xC4, 0xC8, 0x37, 0x10, 0x0C, 0x95,
	0x99, 0x42, 0x50, 0x13, 0x55, 0xB9, 0x62, 0xBC,
	0x53, 0x49, 0x0E, 0xED
};
static unsigned char symtestoutput8[] = {
	0xC9, 0x99, 0xB8, 0x6F, 0x59, 0x52, 0xAD, 0xB5,
	0x22, 0x83, 0xA1, 0x76, 0x48, 0xCD, 0x1E, 0x24,
	0x69, 0xEE, 0x45, 0xBB, 0x12, 0x5F, 0x0B, 0x83,
	0x5A, 0x00, 0x80, 0x19, 0xCF, 0x9B, 0x45, 0xD2,
	0xBF, 0x8A, 0x44, 0x48, 0x68, 0x21, 0x25, 0x45,
	0x9C, 0x2B, 0xD8, 0x1A, 0x4B, 0xE8, 0x25, 0xCB,
	0xF4, 0x9C, 0xF4, 0xFE, 0xE9, 0x9B, 0x72, 0x89,
	0x69, 0x61, 0xB3, 0x0B, 0x15, 0xDC, 0xF4, 0xDC,
	0x2B, 0x6E, 0xDD, 0x2C, 0x43, 0x31, 0xA4, 0xE5,
	0x1A, 0x56, 0x89, 0xA5, 0xC9, 0xCD, 0x83, 0x3E,
	0x62, 0x79, 0xCD, 0xED, 0xE0, 0xA7, 0x57, 0x09,
	0x10, 0x1D, 0x71, 0xEC
};

static unsigned char symtestsboxout8[] = {
	0x65, 0x38, 0xA2, 0xA8, 0xA9, 0x04, 0x05, 0xB1,
	0xE8, 0xA3, 0xEF, 0x3F, 0x86, 0xC1, 0xC7, 0xCE,
	0xE0, 0xA6, 0x32, 0x58, 0xCC, 0x2C, 0xD0, 0x5D,
	0x9C, 0x2D, 0xE5, 0x1C, 0xEC, 0x17, 0xFA, 0x4E,
	0xB0, 0x27, 0xD5, 0xE1, 0x5C, 0xAE, 0x10, 0x28,
	0x5B, 0x35, 0x41, 0xDE, 0x31, 0xB9, 0xDA, 0xE2,
	0x98, 0x74, 0x99, 0xAC, 0xDF, 0x75, 0x9E, 0xEB,
	0xA7, 0x84, 0x11, 0x5E, 0xCB, 0xE6, 0x02, 0xC0,
	0xBF, 0x45, 0x6F, 0xA5, 0x20, 0x0E, 0xCA, 0xD9,
	0x2F, 0xF7, 0x3C, 0xCF, 0x0D, 0x00, 0x60, 0xAA,
	0xD2, 0xC8, 0x72, 0x26, 0x54, 0x19, 0x94, 0x70,
	0xBB, 0x83, 0x62, 0x9A, 0x42, 0x4D, 0x5F, 0x01,
	0x92, 0xDC, 0xCD, 0xE3, 0xA4, 0x0C, 0xBC, 0x57,
	0x73, 0x07, 0x85, 0x47, 0x8E, 0x16, 0xE4, 0x0B,
	0x7E, 0x96, 0xED, 0x2E, 0x23, 0x8B, 0xD8, 0x8A,
	0x12, 0x7D, 0xD6, 0xF2, 0x8F, 0x1B, 0xF6, 0xFC,
	0x3A, 0x33, 0x4F, 0x77, 0x79, 0x4C, 0x91, 0xBD,
	0x52, 0x36, 0xC9, 0xD1, 0x13, 0x6D, 0xA1, 0x66,
	0x9B, 0xF1, 0x06, 0xB3, 0xF8, 0xFB, 0x48, 0xF0,
	0xC5, 0x46, 0x90, 0x63, 0xB8, 0x1F, 0xB2, 0x8C,
	0xBA, 0xD7, 0x69, 0x34, 0x29, 0xE9, 0xEE, 0xC6,
	0x81, 0xAF, 0x3B, 0x30, 0xE7, 0x6E, 0x71, 0xF9,
	0x0F, 0xFD, 0x1A, 0x39, 0xA0, 0xF3, 0x18, 0x1E,
	0x56, 0x43, 0x55, 0x53, 0x40, 0x03, 0xD3, 0xC4,
	0x9D, 0x88, 0xDD, 0x2B, 0x4B, 0x67, 0x93, 0x59,
	0x1D, 0x7C, 0x7F, 0x51, 0xF4, 0x44, 0x61, 0x21,
	0xDB, 0x68, 0xAB, 0x7A, 0x78, 0x14, 0x3E, 0xFE,
	0xBE, 0xAD, 0x7B, 0x89, 0x5A, 0x37, 0x15, 0xB4,
	0xF5, 0xB5, 0x09, 0xD4, 0x3D, 0x82, 0x76, 0xFF,
	0x6C, 0x25, 0x50, 0xB6, 0x80, 0x9F, 0xC2, 0x97,
	0x49, 0x22, 0xEA, 0x6B, 0xB7, 0x08, 0x95, 0x87,
	0x4A, 0x24, 0x64, 0xC3, 0x0A, 0x8D, 0x2A, 0x6A,
	0x5D, 0xC8, 0x4D
};

static unsigned char symtestkey9[] = {	/* 3FE9494B6757073C_8977730CA0054169_B37C98D8C9355719 */
	0x3F, 0xE9, 0x49, 0x4B, 0x67, 0x57, 0x07, 0x3C,
	0x89, 0x77, 0x73, 0x0C, 0xA0, 0x05, 0x41, 0x69,
	0xB3, 0x7C, 0x98, 0xD8, 0xC9, 0x35, 0x57, 0x19
};

static unsigned char symtestiv9[] = {
	0x27, 0x6D, 0xFD, 0x6D, 0x65, 0x0D, 0xD4, 0x22
};

static unsigned char symtestinput9[] = {	/* 2F6757ED8ED5B3DD_FE932AE10DA5DBBE ... */
	0x2F, 0x67, 0x57, 0xED, 0x8E, 0xD5, 0xB3, 0xDD,
	0xFE, 0x93, 0x2A, 0xE1, 0x0D, 0xA5, 0xDB, 0xBE,
	0x90, 0x2C, 0xC4, 0xA9, 0x62, 0xC8, 0x89, 0x7C,
	0xAC, 0x52, 0x18, 0x12, 0xF0, 0x91, 0xD2, 0x1A,
	0x05, 0xB7, 0xEF, 0x39, 0x4D, 0x61, 0x35, 0xF2,
	0x11, 0x89, 0xC3, 0xEA, 0xD2, 0x82, 0x9D, 0x4A,
	0xCC, 0x8D, 0xCD, 0xDA, 0xCB, 0x58, 0x9C, 0x1F,
	0xF3, 0x1E, 0xCC, 0xD5, 0x30, 0xC0, 0x97, 0xEA
};

static unsigned char symtestoutput9[] = {
	0x79, 0x40, 0x53, 0xDB, 0x82, 0x7D, 0xD2, 0xF3,
	0x1B, 0xA7, 0x48, 0x5C, 0x2E, 0x3E, 0x95, 0x38,
	0xDC, 0x03, 0xBF, 0xA2, 0x3C, 0x44, 0xC2, 0xFE,
	0xBF, 0x83, 0x59, 0x02, 0xE8, 0x5F, 0xBA, 0x93,
	0x56, 0x52, 0x8F, 0x85, 0x8D, 0x4C, 0xA7, 0x8A,
	0x00, 0xFC, 0x5D, 0x2B, 0x78, 0x00, 0xFC, 0xC1,
	0x78, 0xC3, 0x29, 0xBF, 0x0F, 0x78, 0xE3, 0x35,
	0xBB, 0x1B, 0xA2, 0x73, 0x57, 0x52, 0x5C, 0x2A
};

static unsigned char symtestkey10[] = {
	0xba, 0x19, 0x37, 0x57, 0x8f, 0x25, 0x0a, 0x61,
	0x98, 0x20, 0x1f, 0xfe, 0x98, 0xa2, 0xd5, 0xdc
};

static unsigned char symtestctr10[] = {
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
};

static unsigned char symtestinput10[] = {
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
};

static unsigned char symtestoutput10[] = {
	0x2b, 0x48, 0x5a, 0xa4, 0xc9, 0xae, 0xa8, 0x21,
	0x8d, 0x7f, 0x17, 0x13, 0x8c, 0xe1, 0x08, 0xe9,
	0x59, 0xc6, 0xef, 0xbb, 0x81, 0xa1, 0x22, 0x8a,
	0x82, 0x38, 0x17, 0x33, 0x48, 0xf3, 0x72, 0x25,
	0xf1, 0xa1, 0x89, 0xf9, 0x97, 0xb2, 0x63, 0xfc,
	0xf3, 0x56, 0xca, 0x68, 0x6e, 0x99, 0x73, 0xb4,
	0xa2, 0x57, 0x7d, 0xf1, 0xe6, 0x90, 0xf2, 0x26,
	0x41, 0xd9, 0x81, 0xd9, 0xc9, 0x51, 0x07, 0xf2
};

static unsigned char symtestkeyscc[] = {
#if 1
	/*
	 * The default key : FE01FE01FE01FE01_01FE01FE01FE01FE_FEFE0101FEFE0101
	 */
	0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01,
	0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE,
	0xFE, 0xFE, 0x01, 0x01, 0xFE, 0xFE, 0x01, 0x01,
#else
	/*
	 * The inverse key : 01FE01FE01FE01FE_FE01FE01FE01FE01_0101FEFE0101FEFE
	 */
	0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE,
	0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01, 0xFE, 0x01,
	0x01, 0x01, 0xFE, 0xFE, 0x01, 0x01, 0xFE, 0xFE
#endif
};

static unsigned char symtestinputscc[] = {
	0xb2, 0x6d, 0xc0, 0x1a, 0x21, 0xdc, 0x49, 0xf2
};

static unsigned char symtestivscc[] = {
	0x27, 0x6d, 0xfd, 0x6d, 0x65, 0x0d, 0xd4, 0x22,
};

static unsigned char symtestoutputscc[] = {
	/* ECB output    0x7a, 0xba, 0x87, 0x61, 0x41, 0xc8, 0xeb, 0x09,  */
	0xF9, 0xCC, 0x3F, 0x35, 0xBB, 0x13, 0xD6, 0x5E,
};

static unsigned char symtestoutputivscc[] = {
	0xB2, 0x6D, 0xC0, 0x1A, 0x21, 0xDC, 0x49, 0xF2,
};

static unsigned char symtestoutputecbscc[] = {
	0x7a, 0xba, 0x87, 0x61, 0x41, 0xc8, 0xeb, 0x09,
};

SYMTESTTYPE symTest[] = {
	{			/* AES CTR Encrypt Test */
	 DPD_AES_CTR_ENC,
	 16,			/* key len */
	 symtestkey1,		/* key */
	 symtestctr1,		/* Initial Counter */
	 FSL_CTR_MOD_72,	/* ctr mod */
	 48,			/* data len */
	 symtestinput1,
	 FSL_RETURN_OK_S,
	 symtestoutput1,
	 NULL,
	 "AES CTR Encrypt Test"},
	{			/* AES CTR Encrypt Test 2 */
	 DPD_AES_CTR_ENC,
	 16,			/* key len */
	 symtestkey1,		/* key */
	 symtestctr1,		/* Initial Counter */
	 FSL_CTR_MOD_72,	/* ctr mod */
	 33,			/* data len */
	 symtestinput1,
	 FSL_RETURN_OK_S,
	 symtestoutput1,
	 NULL,
	 "AES CTR Encrypt Test 2"},
	{			/* AES ECB Test */
	 DPD_AES_ECB_ENC,
	 16,			/* key len */
	 symtestkey2,		/* key */
	 NULL,			/* iv */
	 0,			/* ctr mod */
	 1024,			/* data len */
	 symtestinput2,
	 FSL_RETURN_OK_S,
	 symtestoutput2,
	 NULL,
	 "AES ECB Encrypt Test"},
	{			/* AES CBC Test */
	 DPD_AES_CBC_DEC,
	 16,			/* key len */
	 symtestkey3,		/* key */
	 NULL,			/* iv */
	 0,			/* ctr mod */
	 64,			/* data len */
	 symtestinput3,
	 FSL_RETURN_OK_S,
	 symtestoutput3,
	 NULL,
	 "AES CBC Decrypt Test"},
	{			/* AES CBC Test */
	 DPD_AES_CBC_ENC,
	 16,			/* key len */
	 symtestkey3,		/* key */
	 NULL,			/* iv */
	 0,			/* ctr mod */
	 64,			/* data len */
	 symtestoutput3,
	 FSL_RETURN_OK_S,
	 symtestinput3,
	 NULL,
	 "AES CBC Encrypt Test"},
	{			/* DES ECB Test */
	 DPD_DES_ECB_DEC,
	 8,			/* key len */
	 symtestkey4,		/* key */
	 NULL,			/* iv */
	 0,			/* ctr mod */
	 192,			/* data len */
	 symtestinput4,
	 FSL_RETURN_OK_S,
	 symtestoutput4,
	 NULL,
	 "DES ECB Decrypt Test"},
	{			/* DES Key Parity Test */
	 DPD_DES_KEY_PARITY,
	 8,			/* key len */
	 (unsigned char *)"abcdefgh",	/* key */
	 NULL,			/* iv */
	 0,			/* ctr mod */
	 192,			/* data len */
	 symtestinput4,
	 FSL_RETURN_BAD_KEY_PARITY_S,
	 symtestoutput4,
	 NULL,
	 "DES ECB Key Parity Test"},
	{			/* 3DES ECB Test */
	 DPD_3DES_ECB_DEC,
	 sizeof(symtestkey5),	/* key len */
	 symtestkey5,		/* key */
	 NULL,			/* iv */
	 0,			/* ctr mod */
	 64,			/* data len */
	 symtestinput5,
	 FSL_RETURN_OK_S,
	 symtestoutput5,
	 NULL,
	 "3DES ECB Decrypt Test"}
	,
	{			/* 3DES Bad Key Length Test */
	 DPD_3DES_ECB_DEC,
	 6,			/* key len */
	 symtestkey5,		/* key */
	 NULL,			/* iv */
	 0,			/* ctr mod */
	 64,			/* data len */
	 symtestinput5,
	 FSL_RETURN_BAD_KEY_LENGTH_S,
	 symtestoutput5,
	 NULL,
	 "3DES Bad Key Length Test"}
	,
	{			/* 3DES Bad Data Length Test */
	 DPD_3DES_ECB_DEC,
	 16,			/* key len */
	 symtestkey5,		/* key */
	 NULL,			/* iv */
	 0,			/* ctr mod */
	 63,			/* data len */
	 symtestinput5,
	 FSL_RETURN_BAD_DATA_LENGTH_S,
	 symtestoutput5,
	 NULL,
	 "3DES Bad Data Length Test"}
	,
	{			/* ARC4 'encrypt' */
	 DPD_ARC4_ENC_KEY,
	 4,			/* key len */
	 symtestkey6,		/* key */
	 NULL,			/* iv */
	 0,			/* ctr mod */
	 92,			/* data len */
	 symtestinput6,
	 FSL_RETURN_OK_S,
	 symtestoutput6,
	 NULL,
	 "ARC4 from Key Test"}
	,
	{			/* ARC4 'encrypt' */
	 DPD_ARC4_ENC_KEY,
	 0,			/* key len */
	 NULL,			/* key */
	 symtestsbox7,		/* sbox+ptrs */
	 0,			/* ctr mod */
	 46,			/* data len */
	 symtestinput7,
	 FSL_RETURN_OK_S,
	 symtestoutput7,
	 NULL,
	 "ARC4 from Sbox Test"}
	,
	{			/* ARC4 'encrypt' */
	 DPD_ARC4_ENC_KEY,
	 4,			/* key len */
	 symtestkey8,		/* key */
	 NULL,			/* iv */
	 0,			/* ctr mod */
	 92,			/* data len */
	 symtestinput8,
	 FSL_RETURN_OK_S,
	 symtestoutput8,
	 symtestsboxout8,
	 "ARC4 from Key Test & check SBox"}
	,
	{			/* TDES 3-key CBC */
	 DPD_3DES_CBC_DEC,
	 24,			/* key len */
	 symtestkey9,		/* key */
	 symtestiv9,		/* IV */
	 0,			/* ctr mod */
	 64,			/* data len */
	 symtestinput9,
	 FSL_RETURN_OK_S,
	 symtestoutput9,
	 NULL,
	 "3DES 3-key CBC Decrypt"}
	,
	{			/* TDES 3-key CBC */
	 DPD_3DES_CBC_ENC,
	 24,			/* key len */
	 symtestkey9,		/* key */
	 symtestiv9,		/* IV */
	 0,			/* ctr mod */
	 64,			/* data len */
	 symtestoutput9,
	 FSL_RETURN_OK_S,
	 symtestinput9,
	 NULL,
	 "3DES 3-key CBC Encrypt"}
	,
	{			/* AES CTR Decrypt Test */
	 DPD_AES_CTR_DEC,
	 16,			/* key len */
	 symtestkey10,		/* key */
	 symtestctr10,		/* Initial Counter */
	 FSL_CTR_MOD_16,	/* ctr mod */
	 64,			/* data len */
	 symtestinput10,
	 FSL_RETURN_OK_S,
	 symtestoutput10,
	 NULL,
	 "AES CTR Decrypt Test"}
	,
	{			/* AES CTR Decrypt Test 2 */
	 DPD_AES_CTR_DEC,
	 16,			/* key len */
	 symtestkey10,		/* key */
	 symtestctr10,		/* Initial Counter */
	 FSL_CTR_MOD_16,	/* ctr mod */
	 23,			/* data len */
	 symtestinput10,
	 FSL_RETURN_OK_S,
	 symtestoutput10,
	 NULL,
	 "AES CTR Decrypt Test 2"}
	,
	{			/* SCC Default-key CBC decrypt Test 1 */
	 DPD_3DES_CBC_DEC,
	 24,			/* key len */
	 symtestkeyscc,		/* key */
	 symtestivscc,
	 FSL_CTR_MOD_16,	/* ctr mod */
	 sizeof(symtestinputscc),	/* data len */
	 symtestinputscc,
	 FSL_RETURN_OK_S,
	 symtestoutputscc,
	 symtestoutputivscc,
	 "SCC CBC Decrypt Test 1"}
	,
	{			/* SCC Default-key ECB Encrypt Test 1 */
	 DPD_3DES_ECB_ENC,
	 24,			/* key len */
	 symtestkeyscc,		/* key */
	 NULL,
	 FSL_CTR_MOD_16,	/* ctr mod */
	 sizeof(symtestinputscc),	/* data len */
	 symtestinputscc,
	 FSL_RETURN_OK_S,
	 symtestoutputecbscc,
	 NULL,
	 "SCC ECB Encrypt Test 1"}
	,
};

const unsigned NUM_SYMTESTS = sizeof(symTest) / sizeof(SYMTESTTYPE);

int run_sym_test(fsl_shw_uco_t *, SYMTESTTYPE *);

int run_sym_test(fsl_shw_uco_t * my_ctx, SYMTESTTYPE * test)
{
	fsl_shw_return_t code;
	fsl_shw_sko_t *key_info = malloc(sizeof(*key_info));
	fsl_shw_scco_t *sym_ctx = malloc(sizeof(*sym_ctx));
	fsl_shw_key_alg_t algorithm;
	fsl_shw_sym_mode_t mode;
	fsl_shw_pco_t *cap = fsl_shw_get_capabilities(my_ctx);
	unsigned encrypt;
	unsigned blocksize;
	unsigned contextsize;
	unsigned char *inputText = malloc(test->TextLen);
	unsigned char *outputText = malloc(test->TextLen);
	unsigned char *output_iv = malloc(259);
	int key_established = 0;	/* boolean */
	int ignore_des_key_parity = 1;
	int passed = 0;
	int skipped = 0;

	printf("%s\n", test->testDesc);

	if ((key_info == NULL) || (sym_ctx == NULL) || (inputText == NULL)
	    || (outputText == NULL) || (output_iv == NULL)) {
		printf("Skipping due to memory allocation problems");
		goto out;
	}

	memcpy(inputText, test->pInputText, test->TextLen);

	memset(key_info, 0xcc, sizeof(*key_info));
	memset(sym_ctx, 0x42, sizeof(*sym_ctx));

	switch (test->opId) {
	case DPD_AES_CTR_ENC:
		algorithm = FSL_KEY_ALG_AES;
		blocksize = 16;
		contextsize = 16;
		mode = FSL_SYM_MODE_CTR;
		encrypt = 1;
		break;
	case DPD_AES_CBC_ENC:
		algorithm = FSL_KEY_ALG_AES;
		blocksize = 16;
		contextsize = 16;
		mode = FSL_SYM_MODE_CBC;
		encrypt = 1;
		break;
	case DPD_AES_ECB_ENC:
		algorithm = FSL_KEY_ALG_AES;
		blocksize = 16;
		contextsize = 16;
		mode = FSL_SYM_MODE_ECB;
		encrypt = 1;
		break;
	case DPD_AES_CTR_DEC:
		algorithm = FSL_KEY_ALG_AES;
		blocksize = 16;
		contextsize = 16;
		mode = FSL_SYM_MODE_CTR;
		encrypt = 0;
		break;
	case DPD_AES_CBC_DEC:
		algorithm = FSL_KEY_ALG_AES;
		blocksize = 16;
		contextsize = 16;
		mode = FSL_SYM_MODE_CBC;
		encrypt = 0;
		break;
	case DPD_AES_ECB_DEC:
		algorithm = FSL_KEY_ALG_AES;
		blocksize = 16;
		contextsize = 16;
		mode = FSL_SYM_MODE_ECB;
		encrypt = 0;
		break;
	case DPD_DES_CBC_ENC:
		algorithm = FSL_KEY_ALG_DES;
		blocksize = 8;
		contextsize = 8;
		mode = FSL_SYM_MODE_CBC;
		encrypt = 1;
		break;
	case DPD_DES_ECB_ENC:
		algorithm = FSL_KEY_ALG_DES;
		blocksize = 8;
		contextsize = 8;
		mode = FSL_SYM_MODE_ECB;
		encrypt = 1;
		break;
	case DPD_DES_CBC_DEC:
		algorithm = FSL_KEY_ALG_DES;
		blocksize = 8;
		contextsize = 8;
		mode = FSL_SYM_MODE_CBC;
		encrypt = 0;
		break;
	case DPD_DES_ECB_DEC:
		algorithm = FSL_KEY_ALG_DES;
		blocksize = 8;
		contextsize = 8;
		mode = FSL_SYM_MODE_ECB;
		encrypt = 0;
		break;
	case DPD_DES_KEY_PARITY:
		ignore_des_key_parity = 0;
		algorithm = FSL_KEY_ALG_DES;
		blocksize = 8;
		contextsize = 8;
		mode = FSL_SYM_MODE_ECB;
		encrypt = 0;
		break;
	case DPD_3DES_CBC_ENC:
		algorithm = FSL_KEY_ALG_TDES;
		blocksize = 8;
		contextsize = 8;
		mode = FSL_SYM_MODE_CBC;
		encrypt = 1;
		break;
	case DPD_3DES_ECB_ENC:
		algorithm = FSL_KEY_ALG_TDES;
		blocksize = 8;
		contextsize = 8;
		mode = FSL_SYM_MODE_ECB;
		encrypt = 1;
		break;
	case DPD_3DES_CBC_DEC:
		algorithm = FSL_KEY_ALG_TDES;
		blocksize = 8;
		contextsize = 8;
		mode = FSL_SYM_MODE_CBC;
		encrypt = 0;
		break;
	case DPD_3DES_ECB_DEC:
		algorithm = FSL_KEY_ALG_TDES;
		blocksize = 8;
		contextsize = 8;
		mode = FSL_SYM_MODE_ECB;
		encrypt = 0;
		break;
	case DPD_3DES_KEY_PARITY:
		ignore_des_key_parity = 0;
		algorithm = FSL_KEY_ALG_TDES;
		blocksize = 8;
		contextsize = 8;
		mode = FSL_SYM_MODE_ECB;
		encrypt = 0;
		break;
	case DPD_ARC4_ENC_KEY:
		algorithm = FSL_KEY_ALG_ARC4;
		blocksize = 0;
		contextsize = 259;
		mode = FSL_SYM_MODE_STREAM;
		encrypt = 1;
		break;
	case DPD_ARC4_ENC_SBOX:
		algorithm = FSL_KEY_ALG_ARC4;
		blocksize = 16;	/* arbitrary */
		contextsize = 259;
		mode = FSL_SYM_MODE_STREAM;
		encrypt = 1;
		break;
	default:
		algorithm = -1;
		mode = -1;
		encrypt = 0;
		blocksize = 0;
		contextsize = 0;
		break;
	}

	if ((cap == NULL)
	    || !fsl_shw_pco_check_sym_supported(cap, algorithm, mode)) {
		skipped = 1;	/* skip this test */
		goto out;
	}

	/* Initialize crypto objects */
	if (!fsl_shw_pco_check_pk_supported(cap)) {
		fsl_shw_sko_init(key_info, algorithm);
		/* Insert the key into the key object */
		fsl_shw_sko_set_key(key_info, test->pKey, test->KeyLen);
	} else {
		/* Have dryice. Operation is different */
		fsl_shw_sko_init_pf_key(key_info, algorithm,
					FSL_SHW_PF_KEY_PRG);
		fsl_shw_sko_set_key_length(key_info, test->KeyLen);
		code =
		    fsl_shw_establish_key(my_ctx, key_info, FSL_KEY_WRAP_ACCEPT,
					  test->pKey);
		if (code != FSL_RETURN_OK_S) {
			if (code == test->result) {
				passed = 1;
			} else {
				printf("Could not establish PRG key: %s\n",
				       fsl_error_string(code));
			}
			goto out;
		}
		key_established = 1;
	}
	fsl_shw_scco_init(sym_ctx, algorithm, mode);

	/* Set DES 'ignore parity' for certain cases. */
	if (ignore_des_key_parity && ((algorithm == FSL_KEY_ALG_DES)
				      || (algorithm == FSL_KEY_ALG_TDES))) {
		fsl_shw_sko_set_flags(key_info, FSL_SKO_KEY_IGNORE_PARITY);
	}

	/* Set up context */
	if ((mode == FSL_SYM_MODE_CTR) || (mode == FSL_SYM_MODE_CBC) ||
	    algorithm == FSL_KEY_ALG_ARC4) {
		if (test->pIV_CTR) {
			fsl_shw_scco_set_flags(sym_ctx, FSL_SYM_CTX_LOAD);
			if (mode == FSL_SYM_MODE_CTR) {
				fsl_shw_scco_set_counter_info(sym_ctx,
							      test->pIV_CTR,
							      test->modulus);
			} else {	/* CBC or ARC4 */
				fsl_shw_scco_set_context(sym_ctx,
							 test->pIV_CTR);
			}
		} else {
			fsl_shw_scco_set_flags(sym_ctx, FSL_SYM_CTX_INIT);
		}
	}

	/* If comparing output context ... have API save it */
	if (test->pOutputIV_CTR != NULL) {
		fsl_shw_scco_set_flags(sym_ctx, FSL_SYM_CTX_SAVE);
	}

	memset(outputText, 0x42, test->TextLen);

	if (encrypt) {
		code = fsl_shw_symmetric_encrypt(my_ctx, key_info,
						 sym_ctx, test->TextLen,
						 inputText, outputText);
	} else {
		code = fsl_shw_symmetric_decrypt(my_ctx, key_info,
						 sym_ctx, test->TextLen,
						 inputText, outputText);
	}

	/* If OK is correct result, and that was returned, check data. */
	if ((code == test->result) && (code == FSL_RETURN_OK_S)) {
		passed = 1;	/* assume for now */

		if (compare_result(test->pOutputText, outputText,
				   test->TextLen, encrypt ?
				   "ciphertext" : "plaintext")) {
			passed = 0;	/* failed check */
		}

		/* Check context out if test has value to compare */
		if (test->pOutputIV_CTR != NULL) {
			char *fieldname;

			if ((algorithm == FSL_KEY_ALG_ARC4)
			    || (mode == FSL_SYM_MODE_CBC)) {
				/* Extract the new IV from in the sym_ctx */
				fsl_shw_scco_get_context(sym_ctx, output_iv);
			} else if (mode == FSL_SYM_MODE_CTR) {
				fsl_shw_ctr_mod_t mod;
				fsl_shw_scco_get_counter_info(sym_ctx,
							      output_iv, &mod);
			}
			/* and now compare */
			if (algorithm == FSL_KEY_ALG_ARC4) {
				fieldname = "SBOX+pointers";
			} else if (mode == FSL_SYM_MODE_CBC) {
				fieldname = "IV";
			} else {
				fieldname = "CTR";
			}
			if (compare_result(test->pOutputIV_CTR, output_iv,
					   contextsize, fieldname)) {
				passed = 0;	/* failed */
			}
		}
	} else {		/* OK was not expected or received */
		if (code == test->result) {
			passed = 1;
		} else {
			passed = 0;
			printf
			    ("fsl_shw_symmetric_%s() returned \"%s\" instead of "
			     "\"%s\"\n", encrypt ? "encrypt" : "decrypt",
			     fsl_error_string(code),
			     fsl_error_string(test->result));
		}
	}

      out:
	if (skipped) {
		printf("%s: Skipped\n\n", test->testDesc);
	} else if (passed) {
		printf("%s: Passed\n\n", test->testDesc);
	} else {
		printf("%s: Failed\n\n", test->testDesc);
	}

	if (key_established) {
		/* Need to release the key */
		code = fsl_shw_release_key(my_ctx, key_info);
		if (code != FSL_RETURN_OK_S) {
			printf("Unable to release the established key: %s\n",
			       fsl_error_string(code));
		}
	}

	if (key_info != NULL) {
		free(key_info);
	}
	if (sym_ctx != NULL) {
		free(sym_ctx);
	}
	if (inputText != NULL) {
		free(inputText);
	}
	if (outputText != NULL) {
		free(outputText);
	}
	if (output_iv != NULL) {
		free(output_iv);
	}

	return (skipped ? SKIPPED : (passed ? PASSED : FAILED));
}

/*!
 * Run a series of tests on Symmetric cryptography routines.
 *
 * @param my_ctx    User context to use
 */
void run_symmetric(fsl_shw_uco_t * my_ctx, uint32_t * total_passed_count,
		   uint32_t * total_failed_count)
{
	int passed_count = 0;
	int failed_count = 0;
	int skipped_count = 0;
	unsigned testno;
	enum test_result result;

	for (testno = 0; testno < NUM_SYMTESTS; testno++) {
		result = run_sym_test(my_ctx, symTest + testno);
		switch (result) {
		case PASSED:
			passed_count++;
			break;
		case SKIPPED:
			skipped_count++;
			break;
		case FAILED:
			failed_count++;
			break;
		default:
			printf("Bad result from test!\n");
		}
	}

	printf
	    ("symmetric: %d tests passed, %d tests failed, %d tests skipped\n\n",
	     passed_count, failed_count, skipped_count);

	*total_passed_count += passed_count;
	*total_failed_count += failed_count;

	return;
}
