/* ----------------------------------------------------------------------   
* Copyright (C) 2010 ARM Limited. All rights reserved.   
*   
* $Date:        15. July 2011  
* $Revision: 	V1.0.10  
*   
* Project: 	    CMSIS DSP Library   
* Title:	    arm_rfft_init_f32.c   
*   
* Description:	RFFT & RIFFT Floating point initialisation function   
*   
* Target Processor: Cortex-M4/Cortex-M3/Cortex-M0
*  
* Version 1.0.10 2011/7/15 
*    Big Endian support added and Merged M0 and M3/M4 Source code.  
*   
* Version 1.0.3 2010/11/29  
*    Re-organized the CMSIS folders and updated documentation.   
*    
* Version 1.0.2 2010/11/11   
*    Documentation updated.    
*   
* Version 1.0.1 2010/10/05    
*    Production release and review comments incorporated.   
*   
* Version 1.0.0 2010/09/20    
*    Production release and review comments incorporated.   
*   
* Version 0.0.7  2010/06/10    
*    Misra-C changes done   
* -------------------------------------------------------------------- */


#include "arm_math.h"

/**   
 * @ingroup groupTransforms   
 */

/**   
 * @addtogroup RFFT_RIFFT   
 * @{   
 */

/**   
* \par   
* Generation of realCoefA array:   
* \par   
* 	n = 1024   
* <pre>for (i = 0; i < n; i++)   
*  {   
*    pATable[2 * i] = 0.5 * (1.0 - sin (2 * PI / (double) (2 * n) * (double) i));   
*    pATable[2 * i + 1] = 0.5 * (-1.0 * cos (2 * PI / (double) (2 * n) * (double) i));   
*  } </pre>   
*/



static const float32_t realCoefA[2048] = {
  0.500000000000000000f, -0.500000000000000000f, 0.498466014862060550f,
  -0.499997645616531370f, 0.496932059526443480f, -0.499990582466125490f,
  0.495398133993148800f, -0.499978810548782350f,
  0.493864238262176510f, -0.499962359666824340f, 0.492330402135849000f,
  -0.499941170215606690f, 0.490796625614166260f, -0.499915301799774170f,
  0.489262968301773070f, -0.499884694814682010f,
  0.487729400396347050f, -0.499849408864974980f, 0.486195921897888180f,
  -0.499809414148330690f, 0.484662592411041260f, -0.499764710664749150f,
  0.483129411935806270f, -0.499715298414230350f,
  0.481596380472183230f, -0.499661177396774290f, 0.480063527822494510f,
  -0.499602377414703370f, 0.478530883789062500f, -0.499538868665695190f,
  0.476998418569564820f, -0.499470651149749760f,
  0.475466161966323850f, -0.499397724866867070f, 0.473934143781661990f,
  -0.499320119619369510f, 0.472402364015579220f, -0.499237775802612300f,
  0.470870882272720340f, -0.499150782823562620f,
  0.469339638948440550f, -0.499059051275253300f, 0.467808693647384640f,
  -0.498962640762329100f, 0.466278046369552610f, -0.498861521482467650f,
  0.464747726917266850f, -0.498755723237991330f,
  0.463217705488204960f, -0.498645216226577760f, 0.461688071489334110f,
  -0.498530030250549320f, 0.460158795118331910f, -0.498410135507583620f,
  0.458629876375198360f, -0.498285561800003050f,
  0.457101345062255860f, -0.498156309127807620f, 0.455573230981826780f,
  -0.498022347688674930f, 0.454045534133911130f, -0.497883707284927370f,
  0.452518254518508910f, -0.497740387916564940f,
  0.450991421937942500f, -0.497592359781265260f, 0.449465066194534300f,
  -0.497439652681350710f, 0.447939187288284300f, -0.497282296419143680f,
  0.446413785219192500f, -0.497120231389999390f,
  0.444888889789581300f, -0.496953487396240230f, 0.443364530801773070f,
  -0.496782064437866210f, 0.441840678453445430f, -0.496605962514877320f,
  0.440317392349243160f, -0.496425211429595950f,
  0.438794672489166260f, -0.496239781379699710f, 0.437272518873214720f,
  -0.496049642562866210f, 0.435750931501388550f, -0.495854884386062620f,
  0.434229999780654910f, -0.495655417442321780f,
  0.432709634304046630f, -0.495451331138610840f, 0.431189924478530880f,
  -0.495242536067962650f, 0.429670870304107670f, -0.495029091835021970f,
  0.428152471780776980f, -0.494810998439788820f,
  0.426634758710861210f, -0.494588255882263180f, 0.425117731094360350f,
  -0.494360834360122680f, 0.423601418733596800f, -0.494128793478012080f,
  0.422085791826248170f, -0.493892073631286620f,
  0.420570939779281620f, -0.493650704622268680f, 0.419056802988052370f,
  -0.493404686450958250f, 0.417543441057205200f, -0.493154048919677730f,
  0.416030853986740110f, -0.492898762226104740f,
  0.414519041776657100f, -0.492638826370239260f, 0.413008064031600950f,
  -0.492374241352081300f, 0.411497890949249270f, -0.492105036973953250f,
  0.409988552331924440f, -0.491831213235855100f,
  0.408480048179626460f, -0.491552740335464480f, 0.406972438097000120f,
  -0.491269648075103760f, 0.405465662479400630f, -0.490981936454772950f,
  0.403959810733795170f, -0.490689605474472050f,
  0.402454853057861330f, -0.490392625331878660f, 0.400950789451599120f,
  -0.490091055631637570f, 0.399447679519653320f, -0.489784896373748780f,
  0.397945523262023930f, -0.489474087953567500f,
  0.396444320678710940f, -0.489158689975738530f, 0.394944071769714360f,
  -0.488838672637939450f, 0.393444836139678960f, -0.488514065742492680f,
  0.391946613788604740f, -0.488184869289398190f,
  0.390449374914169310f, -0.487851053476333620f, 0.388953179121017460f,
  -0.487512677907943730f, 0.387458056211471560f, -0.487169682979583740f,
  0.385963946580886840f, -0.486822128295898440f,
  0.384470939636230470f, -0.486469984054565430f, 0.382979035377502440f,
  -0.486113250255584720f, 0.381488204002380370f, -0.485751956701278690f,
  0.379998475313186650f, -0.485386073589324950f,
  0.378509908914566040f, -0.485015630722045900f, 0.377022475004196170f,
  -0.484640628099441530f, 0.375536203384399410f, -0.484261035919189450f,
  0.374051094055175780f, -0.483876913785934450f,
  0.372567176818847660f, -0.483488231897354130f, 0.371084451675415040f,
  -0.483094990253448490f, 0.369602948427200320f, -0.482697218656539920f,
  0.368122667074203490f, -0.482294887304306030f,
  0.366643607616424560f, -0.481888025999069210f, 0.365165829658508300f,
  -0.481476634740829470f, 0.363689333200454710f, -0.481060713529586790f,
  0.362214088439941410f, -0.480640232563018800f,
  0.360740154981613160f, -0.480215251445770260f, 0.359267532825469970f,
  -0.479785770177841190f, 0.357796221971511840f, -0.479351729154586790f,
  0.356326282024383540f, -0.478913217782974240f,
  0.354857653379440310f, -0.478470176458358760f, 0.353390425443649290f,
  -0.478022634983062740f, 0.351924568414688110f, -0.477570593357086180f,
  0.350460082292556760f, -0.477114051580429080f,
  0.348997026681900020f, -0.476653009653091430f, 0.347535371780395510f,
  -0.476187497377395630f, 0.346075177192687990f, -0.475717514753341670f,
  0.344616413116455080f, -0.475243031978607180f,
  0.343159139156341550f, -0.474764078855514530f, 0.341703325510025020f,
  -0.474280685186386110f, 0.340248972177505490f, -0.473792791366577150f,
  0.338796168565750120f, -0.473300457000732420f,
  0.337344855070114140f, -0.472803652286529540f, 0.335895091295242310f,
  -0.472302407026290890f, 0.334446847438812260f, -0.471796721220016480f,
  0.333000183105468750f, -0.471286594867706300f,
  0.331555068492889400f, -0.470772027969360350f, 0.330111563205718990f,
  -0.470253020524978640f, 0.328669637441635130f, -0.469729602336883540f,
  0.327229350805282590f, -0.469201773405075070f,
  0.325790673494338990f, -0.468669503927230830f, 0.324353635311126710f,
  -0.468132823705673220f, 0.322918236255645750f, -0.467591762542724610f,
  0.321484506130218510f, -0.467046260833740230f,
  0.320052474737167360f, -0.466496407985687260f, 0.318622142076492310f,
  -0.465942144393920900f, 0.317193508148193360f, -0.465383470058441160f,
  0.315766572952270510f, -0.464820444583892820f,
  0.314341396093368530f, -0.464253038167953490f, 0.312917977571487430f,
  -0.463681250810623170f, 0.311496287584304810f, -0.463105112314224240f,
  0.310076385736465450f, -0.462524622678756710f,
  0.308658272027969360f, -0.461939752101898190f, 0.307241976261138920f,
  -0.461350560188293460f, 0.305827468633651730f, -0.460757017135620120f,
  0.304414808750152590f, -0.460159152746200560f,
  0.303003966808319090f, -0.459556937217712400f, 0.301595002412796020f,
  -0.458950400352478030f, 0.300187885761260990f, -0.458339542150497440f,
  0.298782676458358760f, -0.457724362611770630f,
  0.297379344701766970f, -0.457104891538620000f, 0.295977920293807980f,
  -0.456481099128723140f, 0.294578403234481810f, -0.455853015184402470f,
  0.293180853128433230f, -0.455220639705657960f,
  0.291785210371017460f, -0.454584002494812010f, 0.290391564369201660f,
  -0.453943043947219850f, 0.288999855518341060f, -0.453297853469848630f,
  0.287610173225402830f, -0.452648371458053590f,
  0.286222457885742190f, -0.451994657516479490f, 0.284836769104003910f,
  -0.451336652040481570f, 0.283453077077865600f, -0.450674414634704590f,
  0.282071471214294430f, -0.450007945299148560f,
  0.280691891908645630f, -0.449337244033813480f, 0.279314368963241580f,
  -0.448662281036376950f, 0.277938932180404660f, -0.447983115911483760f,
  0.276565581560134890f, -0.447299748659133910f,
  0.275194346904754640f, -0.446612149477005000f, 0.273825198411941530f,
  -0.445920348167419430f, 0.272458195686340330f, -0.445224374532699580f,
  0.271093338727951050f, -0.444524168968200680f,
  0.269730657339096070f, -0.443819820880889890f, 0.268370121717453000f,
  -0.443111270666122440f, 0.267011761665344240f, -0.442398548126220700f,
  0.265655577182769780f, -0.441681683063507080f,
  0.264301627874374390f, -0.440960645675659180f, 0.262949883937835690f,
  -0.440235435962677000f, 0.261600375175476070f, -0.439506113529205320f,
  0.260253131389617920f, -0.438772648572921750f,
  0.258908122777938840f, -0.438035041093826290f, 0.257565379142761230f,
  -0.437293320894241330f, 0.256224930286407470f, -0.436547487974166870f,
  0.254886746406555180f, -0.435797542333602910f,
  0.253550916910171510f, -0.435043483972549440f, 0.252217382192611690f,
  -0.434285342693328860f, 0.250886172056198120f, -0.433523118495941160f,
  0.249557301402091980f, -0.432756811380386350f,
  0.248230814933776860f, -0.431986421346664430f, 0.246906682848930360f,
  -0.431211978197097780f, 0.245584934949874880f, -0.430433481931686400f,
  0.244265571236610410f, -0.429650902748107910f,
  0.242948621511459350f, -0.428864300251007080f, 0.241634100675582890f,
  -0.428073674440383910f, 0.240322008728981020f, -0.427278995513916020f,
  0.239012360572814940f, -0.426480293273925780f,
  0.237705156207084660f, -0.425677597522735600f, 0.236400425434112550f,
  -0.424870878458023070f, 0.235098183155059810f, -0.424060165882110600f,
  0.233798429369926450f, -0.423245459794998170f,
  0.232501193881034850f, -0.422426789999008180f, 0.231206461787223820f,
  -0.421604126691818240f, 0.229914262890815730f, -0.420777499675750730f,
  0.228624612092971800f, -0.419946908950805660f,
  0.227337509393692020f, -0.419112354516983030f, 0.226052969694137570f,
  -0.418273866176605220f, 0.224771007895469670f, -0.417431443929672240f,
  0.223491653800010680f, -0.416585087776184080f,
  0.222214877605438230f, -0.415734797716140750f, 0.220940738916397090f,
  -0.414880603551864620f, 0.219669207930564880f, -0.414022535085678100f,
  0.218400329351425170f, -0.413160532712936400f,
  0.217134088277816770f, -0.412294656038284300f, 0.215870529413223270f,
  -0.411424905061721800f, 0.214609622955322270f, -0.410551249980926510f,
  0.213351413607597350f, -0.409673750400543210f,
  0.212095901370048520f, -0.408792406320571900f, 0.210843101143836980f,
  -0.407907217741012570f, 0.209593027830123900f, -0.407018154859542850f,
  0.208345666527748110f, -0.406125307083129880f,
  0.207101076841354370f, -0.405228585004806520f, 0.205859228968620300f,
  -0.404328078031539920f, 0.204620152711868290f, -0.403423786163330080f,
  0.203383848071098330f, -0.402515679597854610f,
  0.202150344848632810f, -0.401603758335113530f, 0.200919643044471740f,
  -0.400688081979751590f, 0.199691757559776310f, -0.399768620729446410f,
  0.198466703295707700f, -0.398845434188842770f,
  0.197244480252265930f, -0.397918462753295900f, 0.196025103330612180f,
  -0.396987736225128170f, 0.194808602333068850f, -0.396053284406661990f,
  0.193594962358474730f, -0.395115107297897340f,
  0.192384198307991030f, -0.394173204898834230f, 0.191176339983940120f,
  -0.393227607011795040f, 0.189971387386322020f, -0.392278283834457400f,
  0.188769355416297910f, -0.391325294971466060f,
  0.187570258975028990f, -0.390368610620498660f, 0.186374098062515260f,
  -0.389408260583877560f, 0.185180887579917910f, -0.388444244861602780f,
  0.183990627527236940f, -0.387476563453674320f,
  0.182803362607955930f, -0.386505216360092160f, 0.181619063019752500f,
  -0.385530263185501100f, 0.180437773466110230f, -0.384551674127578740f,
  0.179259493947029110f, -0.383569449186325070f,
  0.178084224462509160f, -0.382583618164062500f, 0.176911994814872740f,
  -0.381594210863113400f, 0.175742805004119870f, -0.380601197481155400f,
  0.174576655030250550f, -0.379604607820510860f,
  0.173413574695587160f, -0.378604412078857420f, 0.172253578901290890f,
  -0.377600699663162230f, 0.171096652746200560f, -0.376593410968780520f,
  0.169942826032638550f, -0.375582575798034670f,
  0.168792113661766050f, -0.374568194150924680f, 0.167644515633583070f,
  -0.373550295829772950f, 0.166500031948089600f, -0.372528880834579470f,
  0.165358707308769230f, -0.371503978967666630f,
  0.164220526814460750f, -0.370475560426712040f, 0.163085505366325380f,
  -0.369443655014038090f, 0.161953642964363100f, -0.368408292531967160f,
  0.160824984312057500f, -0.367369443178176880f,
  0.159699499607086180f, -0.366327136754989620f, 0.158577233552932740f,
  -0.365281373262405400f, 0.157458171248435970f, -0.364232182502746580f,
  0.156342327594757080f, -0.363179564476013180f,
  0.155229732394218440f, -0.362123548984527590f, 0.154120370745658870f,
  -0.361064106225967410f, 0.153014272451400760f, -0.360001266002655030f,
  0.151911437511444090f, -0.358935028314590450f,
  0.150811880826950070f, -0.357865422964096070f, 0.149715602397918700f,
  -0.356792420148849490f, 0.148622632026672360f, -0.355716109275817870f,
  0.147532954812049870f, -0.354636400938034060f,
  0.146446615457534790f, -0.353553384542465210f, 0.145363584160804750f,
  -0.352467030286788940f, 0.144283905625343320f, -0.351377367973327640f,
  0.143207564949989320f, -0.350284397602081300f,
  0.142134591937065120f, -0.349188119173049930f, 0.141064971685409550f,
  -0.348088562488555910f, 0.139998748898506160f, -0.346985727548599240f,
  0.138935908675193790f, -0.345879614353179930f,
  0.137876465916633610f, -0.344770282506942750f, 0.136820420622825620f,
  -0.343657672405242920f, 0.135767802596092220f, -0.342541843652725220f,
  0.134718611836433410f, -0.341422766447067260f,
  0.133672863245010380f, -0.340300500392913820f, 0.132630556821823120f,
  -0.339175015687942500f, 0.131591722369194030f, -0.338046342134475710f,
  0.130556344985961910f, -0.336914509534835820f,
  0.129524439573287960f, -0.335779488086700440f, 0.128496021032333370f,
  -0.334641307592391970f, 0.127471104264259340f, -0.333499968051910400f,
  0.126449704170227050f, -0.332355499267578130f,
  0.125431805849075320f, -0.331207901239395140f, 0.124417431652545930f,
  -0.330057173967361450f, 0.123406603932380680f, -0.328903347253799440f,
  0.122399315237998960f, -0.327746421098709110f,
  0.121395580470561980f, -0.326586425304412840f, 0.120395407080650330f,
  -0.325423330068588260f, 0.119398809969425200f, -0.324257194995880130f,
  0.118405789136886600f, -0.323088020086288450f,
  0.117416366934776310f, -0.321915775537490840f, 0.116430543363094330f,
  -0.320740520954132080f, 0.115448333323001860f, -0.319562226533889770f,
  0.114469736814498900f, -0.318380922079086300f,
  0.113494776189327240f, -0.317196637392044070f, 0.112523443996906280f,
  -0.316009372472763060f, 0.111555770039558410f, -0.314819127321243290f,
  0.110591746866703030f, -0.313625901937484740f,
  0.109631389379501340f, -0.312429755926132200f, 0.108674705028533940f,
  -0.311230629682540890f, 0.107721701264381410f, -0.310028612613677980f,
  0.106772392988204960f, -0.308823645114898680f,
  0.105826787650585170f, -0.307615786790847780f, 0.104884892702102660f,
  -0.306405037641525270f, 0.103946708142757420f, -0.305191397666931150f,
  0.103012263774871830f, -0.303974896669387820f,
  0.102081544697284700f, -0.302755534648895260f, 0.101154580712318420f,
  -0.301533311605453490f, 0.100231364369392400f, -0.300308227539062500f,
  0.099311910569667816f, -0.299080342054367070f,
  0.098396234214305878f, -0.297849655151367190f, 0.097484335303306580f,
  -0.296616137027740480f, 0.096576221287250519f, -0.295379847288131710f,
  0.095671907067298889f, -0.294140785932540890f,
  0.094771400094032288f, -0.292898923158645630f, 0.093874707818031311f,
  -0.291654318571090700f, 0.092981837689876556f, -0.290406972169876100f,
  0.092092797160148621f, -0.289156883955001830f,
  0.091207593679428101f, -0.287904083728790280f, 0.090326242148876190f,
  -0.286648571491241460f, 0.089448742568492889f, -0.285390377044677730f,
  0.088575109839439392f, -0.284129470586776730f,
  0.087705351412296295f, -0.282865911722183230f, 0.086839467287063599f,
  -0.281599670648574830f, 0.085977479815483093f, -0.280330777168273930f,
  0.085119381546974182f, -0.279059261083602910f,
  0.084265194833278656f, -0.277785122394561770f, 0.083414919674396515f,
  -0.276508361101150510f, 0.082568563520908356f, -0.275228977203369140f,
  0.081726133823394775f, -0.273947030305862430f,
  0.080887645483016968f, -0.272662490606307980f, 0.080053105950355530f,
  -0.271375387907028200f, 0.079222507774829865f, -0.270085722208023070f,
  0.078395880758762360f, -0.268793523311614990f,
  0.077573217451572418f, -0.267498821020126340f, 0.076754532754421234f,
  -0.266201555728912350f, 0.075939826667308807f, -0.264901816844940190f,
  0.075129114091396332f, -0.263599574565887450f,
  0.074322402477264404f, -0.262294828891754150f, 0.073519699275493622f,
  -0.260987639427185060f, 0.072721004486083984f, -0.259678006172180180f,
  0.071926333010196686f, -0.258365899324417110f,
  0.071135692298412323f, -0.257051378488540650f, 0.070349089801311493f,
  -0.255734413862228390f, 0.069566532969474792f, -0.254415065050125120f,
  0.068788021802902222f, -0.253093332052230830f,
  0.068013571202754974f, -0.251769185066223140f, 0.067243188619613647f,
  -0.250442683696746830f, 0.066476874053478241f, -0.249113827943801880f,
  0.065714649856090546f, -0.247782632708549500f,
  0.064956501126289368f, -0.246449097990989690f, 0.064202457666397095f,
  -0.245113238692283630f, 0.063452512025833130f, -0.243775084614753720f,
  0.062706671655178070f, -0.242434620857238770f,
  0.061964951455593109f, -0.241091892123222350f, 0.061227355152368546f,
  -0.239746883511543270f, 0.060493886470794678f, -0.238399609923362730f,
  0.059764556586742401f, -0.237050101161003110f,
  0.059039369225502014f, -0.235698372125625610f, 0.058318331837654114f,
  -0.234344407916069030f, 0.057601451873779297f, -0.232988253235816960f,
  0.056888736784458160f, -0.231629893183708190f,
  0.056180190294981003f, -0.230269357562065120f, 0.055475823581218719f,
  -0.228906646370887760f, 0.054775636643171310f, -0.227541789412498470f,
  0.054079644381999969f, -0.226174786686897280f,
  0.053387850522994995f, -0.224805667996406560f, 0.052700258791446686f,
  -0.223434418439865110f, 0.052016876637935638f, -0.222061067819595340f,
  0.051337707787752151f, -0.220685631036758420f,
  0.050662767142057419f, -0.219308122992515560f, 0.049992054700851440f,
  -0.217928543686866760f, 0.049325577914714813f, -0.216546908020973210f,
  0.048663340508937836f, -0.215163245797157290f,
  0.048005353659391403f, -0.213777542114257810f, 0.047351621091365814f,
  -0.212389841675758360f, 0.046702146530151367f, -0.211000129580497740f,
  0.046056941151618958f, -0.209608450531959530f,
  0.045416008681058884f, -0.208214774727821350f, 0.044779352843761444f,
  -0.206819161772727970f, 0.044146984815597534f, -0.205421581864356990f,
  0.043518904596567154f, -0.204022079706192020f,
  0.042895123362541199f, -0.202620655298233030f, 0.042275641113519669f,
  -0.201217323541641240f, 0.041660469025373459f, -0.199812099337577820f,
  0.041049610823392868f, -0.198404997587203980f,
  0.040443073958158493f, -0.196996018290519710f, 0.039840862154960632f,
  -0.195585191249847410f, 0.039242979139089584f, -0.194172516465187070f,
  0.038649436086416245f, -0.192758023738861080f,
  0.038060232996940613f, -0.191341713070869450f, 0.037475381046533585f,
  -0.189923599362373350f, 0.036894880235195160f, -0.188503712415695190f,
  0.036318738013505936f, -0.187082037329673770f,
  0.035746958106756210f, -0.185658603906631470f, 0.035179551690816879f,
  -0.184233412146568300f, 0.034616518765687943f, -0.182806491851806640f,
  0.034057866781949997f, -0.181377857923507690f,
  0.033503599464893341f, -0.179947525262832640f, 0.032953724265098572f,
  -0.178515478968620300f, 0.032408244907855988f, -0.177081763744354250f,
  0.031867165118455887f, -0.175646379590034480f,
  0.031330492347478867f, -0.174209341406822200f, 0.030798232182860374f,
  -0.172770664095878600f, 0.030270388349890709f, -0.171330362558364870f,
  0.029746964573860168f, -0.169888436794281010f,
  0.029227968305349350f, -0.168444931507110600f, 0.028713401407003403f,
  -0.166999831795692440f, 0.028203271329402924f, -0.165553152561187740f,
  0.027697581797838211f, -0.164104923605918880f,
  0.027196336537599564f, -0.162655144929885860f, 0.026699542999267578f,
  -0.161203846335411070f, 0.026207204908132553f, -0.159751012921333310f,
  0.025719324126839638f, -0.158296689391136170f,
  0.025235909968614578f, -0.156840875744819640f, 0.024756962433457375f,
  -0.155383571982383730f, 0.024282488971948624f, -0.153924822807312010f,
  0.023812493309378624f, -0.152464613318443300f,
  0.023346979171037674f, -0.151002973318099980f, 0.022885952144861221f,
  -0.149539917707443240f, 0.022429415956139565f, -0.148075446486473080f,
  0.021977374330163002f, -0.146609574556350710f,
  0.021529832854866982f, -0.145142331719398500f, 0.021086793392896652f,
  -0.143673732876777650f, 0.020648263394832611f, -0.142203763127326970f,
  0.020214242860674858f, -0.140732467174530030f,
  0.019784741103649139f, -0.139259845018386840f, 0.019359756261110306f,
  -0.137785911560058590f, 0.018939297646284103f, -0.136310681700706480f,
  0.018523367121815681f, -0.134834155440330510f,
  0.018111966550350189f, -0.133356377482414250f, 0.017705103382468224f,
  -0.131877332925796510f, 0.017302779480814934f, -0.130397051572799680f,
  0.016904998570680618f, -0.128915548324584960f,
  0.016511764377355576f, -0.127432823181152340f, 0.016123080626130104f,
  -0.125948905944824220f, 0.015738952904939651f, -0.124463804066181180f,
  0.015359382145106792f, -0.122977524995803830f,
  0.014984373003244400f, -0.121490091085433960f, 0.014613929204642773f,
  -0.120001509785652160f, 0.014248054474592209f, -0.118511803448200230f,
  0.013886751607060432f, -0.117020979523658750f,
  0.013530024327337742f, -0.115529052913188930f, 0.013177875429391861f,
  -0.114036038517951970f, 0.012830308638513088f, -0.112541958689689640f,
  0.012487327679991722f, -0.111046813428401950f,
  0.012148935347795486f, -0.109550617635250090f, 0.011815134435892105f,
  -0.108053401112556460f, 0.011485928669571877f, -0.106555156409740450f,
  0.011161320842802525f, -0.105055920779705050f,
  0.010841314680874348f, -0.103555686771869660f, 0.010525912046432495f,
  -0.102054484188556670f, 0.010215117596089840f, -0.100552320480346680f,
  0.009908932261168957f, -0.099049203097820282f,
  0.009607359766960144f, -0.097545161843299866f, 0.009310402907431126f,
  -0.096040196716785431f, 0.009018065407872200f, -0.094534330070018768f,
  0.008730349130928516f, -0.093027576804161072f,
  0.008447255939245224f, -0.091519944369792938f, 0.008168790489435196f,
  -0.090011447668075562f, 0.007894953712821007f, -0.088502109050750732f,
  0.007625748869031668f, -0.086991935968399048f,
  0.007361178752034903f, -0.085480943322181702f, 0.007101245224475861f,
  -0.083969146013259888f, 0.006845951545983553f, -0.082456558942794800f,
  0.006595299113541842f, -0.080943197011947632f,
  0.006349290721118450f, -0.079429075121879578f, 0.006107929162681103f,
  -0.077914200723171234f, 0.005871216300874949f, -0.076398596167564392f,
  0.005639153998345137f, -0.074882268905639648f,
  0.005411745049059391f, -0.073365233838558197f, 0.005188991315662861f,
  -0.071847513318061829f, 0.004970894660800695f, -0.070329122245311737f,
  0.004757457878440619f, -0.068810060620307922f,
  0.004548682365566492f, -0.067290350794792175f, 0.004344569984823465f,
  -0.065770015120506287f, 0.004145123064517975f, -0.064249053597450256f,
  0.003950343467295170f, -0.062727488577365875f,
  0.003760232590138912f, -0.061205338686704636f, 0.003574792761355639f,
  -0.059682607650756836f, 0.003394025377929211f, -0.058159314095973969f,
  0.003217932302504778f, -0.056635476648807526f,
  0.003046514932066202f, -0.055111102759838104f, 0.002879775362089276f,
  -0.053586211055517197f, 0.002717714523896575f, -0.052060816437005997f,
  0.002560334512963891f, -0.050534930080175400f,
  0.002407636726275086f, -0.049008570611476898f, 0.002259622327983379f,
  -0.047481749206781387f, 0.002116292715072632f, -0.045954477041959763f,
  0.001977649517357349f, -0.044426776468753815f,
  0.001843693898990750f, -0.042898654937744141f, 0.001714427140541375f,
  -0.041370131075382233f, 0.001589850406162441f, -0.039841219782829285f,
  0.001469964860007167f, -0.038311932235956192f,
  0.001354771666228771f, -0.036782283335924149f, 0.001244271872565150f,
  -0.035252287983894348f, 0.001138466643169522f, -0.033721961081027985f,
  0.001037356909364462f, -0.032191313803195953f,
  0.000940943544264883f, -0.030660368502140045f, 0.000849227537401021f,
  -0.029129132628440857f, 0.000762209703680128f, -0.027597622945904732f,
  0.000679890916217119f, -0.026065852493047714f,
  0.000602271873503923f, -0.024533838033676147f, 0.000529353390447795f,
  -0.023001590743660927f, 0.000461136136436835f, -0.021469129249453545f,
  0.000397620693547651f, -0.019936462864279747f,
  0.000338807702064514f, -0.018403612077236176f, 0.000284697714960203f,
  -0.016870586201548576f, 0.000235291256103665f, -0.015337402001023293f,
  0.000190588747500442f, -0.013804072514176369f,
  0.000150590654811822f, -0.012270614504814148f, 0.000115297327283770f,
  -0.010737040080130100f, 0.000084709099610336f, -0.009203365072607994f,
  0.000058826273743762f, -0.007669602986425161f,
  0.000037649078876711f, -0.006135769188404083f, 0.000021177724192967f,
  -0.004601877182722092f, 0.000009412358849659f, -0.003067942336201668f,
  0.000002353095169383f, -0.001533978385850787f,
  0.000000000000000000f, -0.000000000000023345f, 0.000002353095169383f,
  0.001533978385850787f, 0.000009412358849659f, 0.003067942336201668f,
  0.000021177724192967f, 0.004601877182722092f,
  0.000037649078876711f, 0.006135769188404083f, 0.000058826273743762f,
  0.007669602986425161f, 0.000084709099610336f, 0.009203365072607994f,
  0.000115297327283770f, 0.010737040080130100f,
  0.000150590654811822f, 0.012270614504814148f, 0.000190588747500442f,
  0.013804072514176369f, 0.000235291256103665f, 0.015337402001023293f,
  0.000284697714960203f, 0.016870586201548576f,
  0.000338807702064514f, 0.018403612077236176f, 0.000397620693547651f,
  0.019936462864279747f, 0.000461136136436835f, 0.021469129249453545f,
  0.000529353390447795f, 0.023001590743660927f,
  0.000602271873503923f, 0.024533838033676147f, 0.000679890916217119f,
  0.026065852493047714f, 0.000762209703680128f, 0.027597622945904732f,
  0.000849227537401021f, 0.029129132628440857f,
  0.000940943544264883f, 0.030660368502140045f, 0.001037356909364462f,
  0.032191313803195953f, 0.001138466643169522f, 0.033721961081027985f,
  0.001244271872565150f, 0.035252287983894348f,
  0.001354771666228771f, 0.036782283335924149f, 0.001469964860007167f,
  0.038311932235956192f, 0.001589850406162441f, 0.039841219782829285f,
  0.001714427140541375f, 0.041370131075382233f,
  0.001843693898990750f, 0.042898654937744141f, 0.001977649517357349f,
  0.044426776468753815f, 0.002116292715072632f, 0.045954477041959763f,
  0.002259622327983379f, 0.047481749206781387f,
  0.002407636726275086f, 0.049008570611476898f, 0.002560334512963891f,
  0.050534930080175400f, 0.002717714523896575f, 0.052060816437005997f,
  0.002879775362089276f, 0.053586211055517197f,
  0.003046514932066202f, 0.055111102759838104f, 0.003217932302504778f,
  0.056635476648807526f, 0.003394025377929211f, 0.058159314095973969f,
  0.003574792761355639f, 0.059682607650756836f,
  0.003760232590138912f, 0.061205338686704636f, 0.003950343467295170f,
  0.062727488577365875f, 0.004145123064517975f, 0.064249053597450256f,
  0.004344569984823465f, 0.065770015120506287f,
  0.004548682365566492f, 0.067290350794792175f, 0.004757457878440619f,
  0.068810060620307922f, 0.004970894660800695f, 0.070329122245311737f,
  0.005188991315662861f, 0.071847513318061829f,
  0.005411745049059391f, 0.073365233838558197f, 0.005639153998345137f,
  0.074882268905639648f, 0.005871216300874949f, 0.076398596167564392f,
  0.006107929162681103f, 0.077914200723171234f,
  0.006349290721118450f, 0.079429075121879578f, 0.006595299113541842f,
  0.080943197011947632f, 0.006845951545983553f, 0.082456558942794800f,
  0.007101245224475861f, 0.083969146013259888f,
  0.007361178752034903f, 0.085480943322181702f, 0.007625748869031668f,
  0.086991935968399048f, 0.007894953712821007f, 0.088502109050750732f,
  0.008168790489435196f, 0.090011447668075562f,
  0.008447255939245224f, 0.091519944369792938f, 0.008730349130928516f,
  0.093027576804161072f, 0.009018065407872200f, 0.094534330070018768f,
  0.009310402907431126f, 0.096040196716785431f,
  0.009607359766960144f, 0.097545161843299866f, 0.009908932261168957f,
  0.099049203097820282f, 0.010215117596089840f, 0.100552320480346680f,
  0.010525912046432495f, 0.102054484188556670f,
  0.010841314680874348f, 0.103555686771869660f, 0.011161320842802525f,
  0.105055920779705050f, 0.011485928669571877f, 0.106555156409740450f,
  0.011815134435892105f, 0.108053401112556460f,
  0.012148935347795486f, 0.109550617635250090f, 0.012487327679991722f,
  0.111046813428401950f, 0.012830308638513088f, 0.112541958689689640f,
  0.013177875429391861f, 0.114036038517951970f,
  0.013530024327337742f, 0.115529052913188930f, 0.013886751607060432f,
  0.117020979523658750f, 0.014248054474592209f, 0.118511803448200230f,
  0.014613929204642773f, 0.120001509785652160f,
  0.014984373003244400f, 0.121490091085433960f, 0.015359382145106792f,
  0.122977524995803830f, 0.015738952904939651f, 0.124463804066181180f,
  0.016123080626130104f, 0.125948905944824220f,
  0.016511764377355576f, 0.127432823181152340f, 0.016904998570680618f,
  0.128915548324584960f, 0.017302779480814934f, 0.130397051572799680f,
  0.017705103382468224f, 0.131877332925796510f,
  0.018111966550350189f, 0.133356377482414250f, 0.018523367121815681f,
  0.134834155440330510f, 0.018939297646284103f, 0.136310681700706480f,
  0.019359756261110306f, 0.137785911560058590f,
  0.019784741103649139f, 0.139259845018386840f, 0.020214242860674858f,
  0.140732467174530030f, 0.020648263394832611f, 0.142203763127326970f,
  0.021086793392896652f, 0.143673732876777650f,
  0.021529832854866982f, 0.145142331719398500f, 0.021977374330163002f,
  0.146609574556350710f, 0.022429415956139565f, 0.148075446486473080f,
  0.022885952144861221f, 0.149539917707443240f,
  0.023346979171037674f, 0.151002973318099980f, 0.023812493309378624f,
  0.152464613318443300f, 0.024282488971948624f, 0.153924822807312010f,
  0.024756962433457375f, 0.155383571982383730f,
  0.025235909968614578f, 0.156840875744819640f, 0.025719324126839638f,
  0.158296689391136170f, 0.026207204908132553f, 0.159751012921333310f,
  0.026699542999267578f, 0.161203846335411070f,
  0.027196336537599564f, 0.162655144929885860f, 0.027697581797838211f,
  0.164104923605918880f, 0.028203271329402924f, 0.165553152561187740f,
  0.028713401407003403f, 0.166999831795692440f,
  0.029227968305349350f, 0.168444931507110600f, 0.029746964573860168f,
  0.169888436794281010f, 0.030270388349890709f, 0.171330362558364870f,
  0.030798232182860374f, 0.172770664095878600f,
  0.031330492347478867f, 0.174209341406822200f, 0.031867165118455887f,
  0.175646379590034480f, 0.032408244907855988f, 0.177081763744354250f,
  0.032953724265098572f, 0.178515478968620300f,
  0.033503599464893341f, 0.179947525262832640f, 0.034057866781949997f,
  0.181377857923507690f, 0.034616518765687943f, 0.182806491851806640f,
  0.035179551690816879f, 0.184233412146568300f,
  0.035746958106756210f, 0.185658603906631470f, 0.036318738013505936f,
  0.187082037329673770f, 0.036894880235195160f, 0.188503712415695190f,
  0.037475381046533585f, 0.189923599362373350f,
  0.038060232996940613f, 0.191341713070869450f, 0.038649436086416245f,
  0.192758023738861080f, 0.039242979139089584f, 0.194172516465187070f,
  0.039840862154960632f, 0.195585191249847410f,
  0.040443073958158493f, 0.196996018290519710f, 0.041049610823392868f,
  0.198404997587203980f, 0.041660469025373459f, 0.199812099337577820f,
  0.042275641113519669f, 0.201217323541641240f,
  0.042895123362541199f, 0.202620655298233030f, 0.043518904596567154f,
  0.204022079706192020f, 0.044146984815597534f, 0.205421581864356990f,
  0.044779352843761444f, 0.206819161772727970f,
  0.045416008681058884f, 0.208214774727821350f, 0.046056941151618958f,
  0.209608450531959530f, 0.046702146530151367f, 0.211000129580497740f,
  0.047351621091365814f, 0.212389841675758360f,
  0.048005353659391403f, 0.213777542114257810f, 0.048663340508937836f,
  0.215163245797157290f, 0.049325577914714813f, 0.216546908020973210f,
  0.049992054700851440f, 0.217928543686866760f,
  0.050662767142057419f, 0.219308122992515560f, 0.051337707787752151f,
  0.220685631036758420f, 0.052016876637935638f, 0.222061067819595340f,
  0.052700258791446686f, 0.223434418439865110f,
  0.053387850522994995f, 0.224805667996406560f, 0.054079644381999969f,
  0.226174786686897280f, 0.054775636643171310f, 0.227541789412498470f,
  0.055475823581218719f, 0.228906646370887760f,
  0.056180190294981003f, 0.230269357562065120f, 0.056888736784458160f,
  0.231629893183708190f, 0.057601451873779297f, 0.232988253235816960f,
  0.058318331837654114f, 0.234344407916069030f,
  0.059039369225502014f, 0.235698372125625610f, 0.059764556586742401f,
  0.237050101161003110f, 0.060493886470794678f, 0.238399609923362730f,
  0.061227355152368546f, 0.239746883511543270f,
  0.061964951455593109f, 0.241091892123222350f, 0.062706671655178070f,
  0.242434620857238770f, 0.063452512025833130f, 0.243775084614753720f,
  0.064202457666397095f, 0.245113238692283630f,
  0.064956501126289368f, 0.246449097990989690f, 0.065714649856090546f,
  0.247782632708549500f, 0.066476874053478241f, 0.249113827943801880f,
  0.067243188619613647f, 0.250442683696746830f,
  0.068013571202754974f, 0.251769185066223140f, 0.068788021802902222f,
  0.253093332052230830f, 0.069566532969474792f, 0.254415065050125120f,
  0.070349089801311493f, 0.255734413862228390f,
  0.071135692298412323f, 0.257051378488540650f, 0.071926333010196686f,
  0.258365899324417110f, 0.072721004486083984f, 0.259678006172180180f,
  0.073519699275493622f, 0.260987639427185060f,
  0.074322402477264404f, 0.262294828891754150f, 0.075129114091396332f,
  0.263599574565887450f, 0.075939826667308807f, 0.264901816844940190f,
  0.076754532754421234f, 0.266201555728912350f,
  0.077573217451572418f, 0.267498821020126340f, 0.078395880758762360f,
  0.268793523311614990f, 0.079222507774829865f, 0.270085722208023070f,
  0.080053105950355530f, 0.271375387907028200f,
  0.080887645483016968f, 0.272662490606307980f, 0.081726133823394775f,
  0.273947030305862430f, 0.082568563520908356f, 0.275228977203369140f,
  0.083414919674396515f, 0.276508361101150510f,
  0.084265194833278656f, 0.277785122394561770f, 0.085119381546974182f,
  0.279059261083602910f, 0.085977479815483093f, 0.280330777168273930f,
  0.086839467287063599f, 0.281599670648574830f,
  0.087705351412296295f, 0.282865911722183230f, 0.088575109839439392f,
  0.284129470586776730f, 0.089448742568492889f, 0.285390377044677730f,
  0.090326242148876190f, 0.286648571491241460f,
  0.091207593679428101f, 0.287904083728790280f, 0.092092797160148621f,
  0.289156883955001830f, 0.092981837689876556f, 0.290406972169876100f,
  0.093874707818031311f, 0.291654318571090700f,
  0.094771400094032288f, 0.292898923158645630f, 0.095671907067298889f,
  0.294140785932540890f, 0.096576221287250519f, 0.295379847288131710f,
  0.097484335303306580f, 0.296616137027740480f,
  0.098396234214305878f, 0.297849655151367190f, 0.099311910569667816f,
  0.299080342054367070f, 0.100231364369392400f, 0.300308227539062500f,
  0.101154580712318420f, 0.301533311605453490f,
  0.102081544697284700f, 0.302755534648895260f, 0.103012263774871830f,
  0.303974896669387820f, 0.103946708142757420f, 0.305191397666931150f,
  0.104884892702102660f, 0.306405037641525270f,
  0.105826787650585170f, 0.307615786790847780f, 0.106772392988204960f,
  0.308823645114898680f, 0.107721701264381410f, 0.310028612613677980f,
  0.108674705028533940f, 0.311230629682540890f,
  0.109631389379501340f, 0.312429755926132200f, 0.110591746866703030f,
  0.313625901937484740f, 0.111555770039558410f, 0.314819127321243290f,
  0.112523443996906280f, 0.316009372472763060f,
  0.113494776189327240f, 0.317196637392044070f, 0.114469736814498900f,
  0.318380922079086300f, 0.115448333323001860f, 0.319562226533889770f,
  0.116430543363094330f, 0.320740520954132080f,
  0.117416366934776310f, 0.321915775537490840f, 0.118405789136886600f,
  0.323088020086288450f, 0.119398809969425200f, 0.324257194995880130f,
  0.120395407080650330f, 0.325423330068588260f,
  0.121395580470561980f, 0.326586425304412840f, 0.122399315237998960f,
  0.327746421098709110f, 0.123406603932380680f, 0.328903347253799440f,
  0.124417431652545930f, 0.330057173967361450f,
  0.125431805849075320f, 0.331207901239395140f, 0.126449704170227050f,
  0.332355499267578130f, 0.127471104264259340f, 0.333499968051910400f,
  0.128496021032333370f, 0.334641307592391970f,
  0.129524439573287960f, 0.335779488086700440f, 0.130556344985961910f,
  0.336914509534835820f, 0.131591722369194030f, 0.338046342134475710f,
  0.132630556821823120f, 0.339175015687942500f,
  0.133672863245010380f, 0.340300500392913820f, 0.134718611836433410f,
  0.341422766447067260f, 0.135767802596092220f, 0.342541843652725220f,
  0.136820420622825620f, 0.343657672405242920f,
  0.137876465916633610f, 0.344770282506942750f, 0.138935908675193790f,
  0.345879614353179930f, 0.139998748898506160f, 0.346985727548599240f,
  0.141064971685409550f, 0.348088562488555910f,
  0.142134591937065120f, 0.349188119173049930f, 0.143207564949989320f,
  0.350284397602081300f, 0.144283905625343320f, 0.351377367973327640f,
  0.145363584160804750f, 0.352467030286788940f,
  0.146446615457534790f, 0.353553384542465210f, 0.147532954812049870f,
  0.354636400938034060f, 0.148622632026672360f, 0.355716109275817870f,
  0.149715602397918700f, 0.356792420148849490f,
  0.150811880826950070f, 0.357865422964096070f, 0.151911437511444090f,
  0.358935028314590450f, 0.153014272451400760f, 0.360001266002655030f,
  0.154120370745658870f, 0.361064106225967410f,
  0.155229732394218440f, 0.362123548984527590f, 0.156342327594757080f,
  0.363179564476013180f, 0.157458171248435970f, 0.364232182502746580f,
  0.158577233552932740f, 0.365281373262405400f,
  0.159699499607086180f, 0.366327136754989620f, 0.160824984312057500f,
  0.367369443178176880f, 0.161953642964363100f, 0.368408292531967160f,
  0.163085505366325380f, 0.369443655014038090f,
  0.164220526814460750f, 0.370475560426712040f, 0.165358707308769230f,
  0.371503978967666630f, 0.166500031948089600f, 0.372528880834579470f,
  0.167644515633583070f, 0.373550295829772950f,
  0.168792113661766050f, 0.374568194150924680f, 0.169942826032638550f,
  0.375582575798034670f, 0.171096652746200560f, 0.376593410968780520f,
  0.172253578901290890f, 0.377600699663162230f,
  0.173413574695587160f, 0.378604412078857420f, 0.174576655030250550f,
  0.379604607820510860f, 0.175742805004119870f, 0.380601197481155400f,
  0.176911994814872740f, 0.381594210863113400f,
  0.178084224462509160f, 0.382583618164062500f, 0.179259493947029110f,
  0.383569449186325070f, 0.180437773466110230f, 0.384551674127578740f,
  0.181619063019752500f, 0.385530263185501100f,
  0.182803362607955930f, 0.386505216360092160f, 0.183990627527236940f,
  0.387476563453674320f, 0.185180887579917910f, 0.388444244861602780f,
  0.186374098062515260f, 0.389408260583877560f,
  0.187570258975028990f, 0.390368610620498660f, 0.188769355416297910f,
  0.391325294971466060f, 0.189971387386322020f, 0.392278283834457400f,
  0.191176339983940120f, 0.393227607011795040f,
  0.192384198307991030f, 0.394173204898834230f, 0.193594962358474730f,
  0.395115107297897340f, 0.194808602333068850f, 0.396053284406661990f,
  0.196025103330612180f, 0.396987736225128170f,
  0.197244480252265930f, 0.397918462753295900f, 0.198466703295707700f,
  0.398845434188842770f, 0.199691757559776310f, 0.399768620729446410f,
  0.200919643044471740f, 0.400688081979751590f,
  0.202150344848632810f, 0.401603758335113530f, 0.203383848071098330f,
  0.402515679597854610f, 0.204620152711868290f, 0.403423786163330080f,
  0.205859228968620300f, 0.404328078031539920f,
  0.207101076841354370f, 0.405228585004806520f, 0.208345666527748110f,
  0.406125307083129880f, 0.209593027830123900f, 0.407018154859542850f,
  0.210843101143836980f, 0.407907217741012570f,
  0.212095901370048520f, 0.408792406320571900f, 0.213351413607597350f,
  0.409673750400543210f, 0.214609622955322270f, 0.410551249980926510f,
  0.215870529413223270f, 0.411424905061721800f,
  0.217134088277816770f, 0.412294656038284300f, 0.218400329351425170f,
  0.413160532712936400f, 0.219669207930564880f, 0.414022535085678100f,
  0.220940738916397090f, 0.414880603551864620f,
  0.222214877605438230f, 0.415734797716140750f, 0.223491653800010680f,
  0.416585087776184080f, 0.224771007895469670f, 0.417431443929672240f,
  0.226052969694137570f, 0.418273866176605220f,
  0.227337509393692020f, 0.419112354516983030f, 0.228624612092971800f,
  0.419946908950805660f, 0.229914262890815730f, 0.420777499675750730f,
  0.231206461787223820f, 0.421604126691818240f,
  0.232501193881034850f, 0.422426789999008180f, 0.233798429369926450f,
  0.423245459794998170f, 0.235098183155059810f, 0.424060165882110600f,
  0.236400425434112550f, 0.424870878458023070f,
  0.237705156207084660f, 0.425677597522735600f, 0.239012360572814940f,
  0.426480293273925780f, 0.240322008728981020f, 0.427278995513916020f,
  0.241634100675582890f, 0.428073674440383910f,
  0.242948621511459350f, 0.428864300251007080f, 0.244265571236610410f,
  0.429650902748107910f, 0.245584934949874880f, 0.430433481931686400f,
  0.246906682848930360f, 0.431211978197097780f,
  0.248230814933776860f, 0.431986421346664430f, 0.249557301402091980f,
  0.432756811380386350f, 0.250886172056198120f, 0.433523118495941160f,
  0.252217382192611690f, 0.434285342693328860f,
  0.253550916910171510f, 0.435043483972549440f, 0.254886746406555180f,
  0.435797542333602910f, 0.256224930286407470f, 0.436547487974166870f,
  0.257565379142761230f, 0.437293320894241330f,
  0.258908122777938840f, 0.438035041093826290f, 0.260253131389617920f,
  0.438772648572921750f, 0.261600375175476070f, 0.439506113529205320f,
  0.262949883937835690f, 0.440235435962677000f,
  0.264301627874374390f, 0.440960645675659180f, 0.265655577182769780f,
  0.441681683063507080f, 0.267011761665344240f, 0.442398548126220700f,
  0.268370121717453000f, 0.443111270666122440f,
  0.269730657339096070f, 0.443819820880889890f, 0.271093338727951050f,
  0.444524168968200680f, 0.272458195686340330f, 0.445224374532699580f,
  0.273825198411941530f, 0.445920348167419430f,
  0.275194346904754640f, 0.446612149477005000f, 0.276565581560134890f,
  0.447299748659133910f, 0.277938932180404660f, 0.447983115911483760f,
  0.279314368963241580f, 0.448662281036376950f,
  0.280691891908645630f, 0.449337244033813480f, 0.282071471214294430f,
  0.450007945299148560f, 0.283453077077865600f, 0.450674414634704590f,
  0.284836769104003910f, 0.451336652040481570f,
  0.286222457885742190f, 0.451994657516479490f, 0.287610173225402830f,
  0.452648371458053590f, 0.288999855518341060f, 0.453297853469848630f,
  0.290391564369201660f, 0.453943043947219850f,
  0.291785210371017460f, 0.454584002494812010f, 0.293180853128433230f,
  0.455220639705657960f, 0.294578403234481810f, 0.455853015184402470f,
  0.295977920293807980f, 0.456481099128723140f,
  0.297379344701766970f, 0.457104891538620000f, 0.298782676458358760f,
  0.457724362611770630f, 0.300187885761260990f, 0.458339542150497440f,
  0.301595002412796020f, 0.458950400352478030f,
  0.303003966808319090f, 0.459556937217712400f, 0.304414808750152590f,
  0.460159152746200560f, 0.305827468633651730f, 0.460757017135620120f,
  0.307241976261138920f, 0.461350560188293460f,
  0.308658272027969360f, 0.461939752101898190f, 0.310076385736465450f,
  0.462524622678756710f, 0.311496287584304810f, 0.463105112314224240f,
  0.312917977571487430f, 0.463681250810623170f,
  0.314341396093368530f, 0.464253038167953490f, 0.315766572952270510f,
  0.464820444583892820f, 0.317193508148193360f, 0.465383470058441160f,
  0.318622142076492310f, 0.465942144393920900f,
  0.320052474737167360f, 0.466496407985687260f, 0.321484506130218510f,
  0.467046260833740230f, 0.322918236255645750f, 0.467591762542724610f,
  0.324353635311126710f, 0.468132823705673220f,
  0.325790673494338990f, 0.468669503927230830f, 0.327229350805282590f,
  0.469201773405075070f, 0.328669637441635130f, 0.469729602336883540f,
  0.330111563205718990f, 0.470253020524978640f,
  0.331555068492889400f, 0.470772027969360350f, 0.333000183105468750f,
  0.471286594867706300f, 0.334446847438812260f, 0.471796721220016480f,
  0.335895091295242310f, 0.472302407026290890f,
  0.337344855070114140f, 0.472803652286529540f, 0.338796168565750120f,
  0.473300457000732420f, 0.340248972177505490f, 0.473792791366577150f,
  0.341703325510025020f, 0.474280685186386110f,
  0.343159139156341550f, 0.474764078855514530f, 0.344616413116455080f,
  0.475243031978607180f, 0.346075177192687990f, 0.475717514753341670f,
  0.347535371780395510f, 0.476187497377395630f,
  0.348997026681900020f, 0.476653009653091430f, 0.350460082292556760f,
  0.477114051580429080f, 0.351924568414688110f, 0.477570593357086180f,
  0.353390425443649290f, 0.478022634983062740f,
  0.354857653379440310f, 0.478470176458358760f, 0.356326282024383540f,
  0.478913217782974240f, 0.357796221971511840f, 0.479351729154586790f,
  0.359267532825469970f, 0.479785770177841190f,
  0.360740154981613160f, 0.480215251445770260f, 0.362214088439941410f,
  0.480640232563018800f, 0.363689333200454710f, 0.481060713529586790f,
  0.365165829658508300f, 0.481476634740829470f,
  0.366643607616424560f, 0.481888025999069210f, 0.368122667074203490f,
  0.482294887304306030f, 0.369602948427200320f, 0.482697218656539920f,
  0.371084451675415040f, 0.483094990253448490f,
  0.372567176818847660f, 0.483488231897354130f, 0.374051094055175780f,
  0.483876913785934450f, 0.375536203384399410f, 0.484261035919189450f,
  0.377022475004196170f, 0.484640628099441530f,
  0.378509908914566040f, 0.485015630722045900f, 0.379998475313186650f,
  0.485386073589324950f, 0.381488204002380370f, 0.485751956701278690f,
  0.382979035377502440f, 0.486113250255584720f,
  0.384470939636230470f, 0.486469984054565430f, 0.385963946580886840f,
  0.486822128295898440f, 0.387458056211471560f, 0.487169682979583740f,
  0.388953179121017460f, 0.487512677907943730f,
  0.390449374914169310f, 0.487851053476333620f, 0.391946613788604740f,
  0.488184869289398190f, 0.393444836139678960f, 0.488514065742492680f,
  0.394944071769714360f, 0.488838672637939450f,
  0.396444320678710940f, 0.489158689975738530f, 0.397945523262023930f,
  0.489474087953567500f, 0.399447679519653320f, 0.489784896373748780f,
  0.400950789451599120f, 0.490091055631637570f,
  0.402454853057861330f, 0.490392625331878660f, 0.403959810733795170f,
  0.490689605474472050f, 0.405465662479400630f, 0.490981936454772950f,
  0.406972438097000120f, 0.491269648075103760f,
  0.408480048179626460f, 0.491552740335464480f, 0.409988552331924440f,
  0.491831213235855100f, 0.411497890949249270f, 0.492105036973953250f,
  0.413008064031600950f, 0.492374241352081300f,
  0.414519041776657100f, 0.492638826370239260f, 0.416030853986740110f,
  0.492898762226104740f, 0.417543441057205200f, 0.493154048919677730f,
  0.419056802988052370f, 0.493404686450958250f,
  0.420570939779281620f, 0.493650704622268680f, 0.422085791826248170f,
  0.493892073631286620f, 0.423601418733596800f, 0.494128793478012080f,
  0.425117731094360350f, 0.494360834360122680f,
  0.426634758710861210f, 0.494588255882263180f, 0.428152471780776980f,
  0.494810998439788820f, 0.429670870304107670f, 0.495029091835021970f,
  0.431189924478530880f, 0.495242536067962650f,
  0.432709634304046630f, 0.495451331138610840f, 0.434229999780654910f,
  0.495655417442321780f, 0.435750931501388550f, 0.495854884386062620f,
  0.437272518873214720f, 0.496049642562866210f,
  0.438794672489166260f, 0.496239781379699710f, 0.440317392349243160f,
  0.496425211429595950f, 0.441840678453445430f, 0.496605962514877320f,
  0.443364530801773070f, 0.496782064437866210f,
  0.444888889789581300f, 0.496953487396240230f, 0.446413785219192500f,
  0.497120231389999390f, 0.447939187288284300f, 0.497282296419143680f,
  0.449465066194534300f, 0.497439652681350710f,
  0.450991421937942500f, 0.497592359781265260f, 0.452518254518508910f,
  0.497740387916564940f, 0.454045534133911130f, 0.497883707284927370f,
  0.455573230981826780f, 0.498022347688674930f,
  0.457101345062255860f, 0.498156309127807620f, 0.458629876375198360f,
  0.498285561800003050f, 0.460158795118331910f, 0.498410135507583620f,
  0.461688071489334110f, 0.498530030250549320f,
  0.463217705488204960f, 0.498645216226577760f, 0.464747726917266850f,
  0.498755723237991330f, 0.466278046369552610f, 0.498861521482467650f,
  0.467808693647384640f, 0.498962640762329100f,
  0.469339638948440550f, 0.499059051275253300f, 0.470870882272720340f,
  0.499150782823562620f, 0.472402364015579220f, 0.499237775802612300f,
  0.473934143781661990f, 0.499320119619369510f,
  0.475466161966323850f, 0.499397724866867070f, 0.476998418569564820f,
  0.499470651149749760f, 0.478530883789062500f, 0.499538868665695190f,
  0.480063527822494510f, 0.499602377414703370f,
  0.481596380472183230f, 0.499661177396774290f, 0.483129411935806270f,
  0.499715298414230350f, 0.484662592411041260f, 0.499764710664749150f,
  0.486195921897888180f, 0.499809414148330690f,
  0.487729400396347050f, 0.499849408864974980f, 0.489262968301773070f,
  0.499884694814682010f, 0.490796625614166260f, 0.499915301799774170f,
  0.492330402135849000f, 0.499941170215606690f,
  0.493864238262176510f, 0.499962359666824340f, 0.495398133993148800f,
  0.499978810548782350f, 0.496932059526443480f, 0.499990582466125490f,
  0.498466014862060550f, 0.499997645616531370f
};


/**   
* \par  
* Generation of realCoefB array:   
* \par  
* n = 1024   
* <pre>for (i = 0; i < n; i++)   
* {   
*    pBTable[2 * i] = 0.5 * (1.0 + sin (2 * PI / (double) (2 * n) * (double) i));   
*    pBTable[2 * i + 1] = 0.5 * (1.0 * cos (2 * PI / (double) (2 * n) * (double) i));   
*  } </pre>   
*   
*/
static const float32_t realCoefB[2048] = {
  0.500000000000000000f, 0.500000000000000000f, 0.501533985137939450f,
  0.499997645616531370f, 0.503067970275878910f, 0.499990582466125490f,
  0.504601895809173580f, 0.499978810548782350f,
  0.506135761737823490f, 0.499962359666824340f, 0.507669627666473390f,
  0.499941170215606690f, 0.509203374385833740f, 0.499915301799774170f,
  0.510737061500549320f, 0.499884694814682010f,
  0.512270629405975340f, 0.499849408864974980f, 0.513804078102111820f,
  0.499809414148330690f, 0.515337407588958740f, 0.499764710664749150f,
  0.516870558261871340f, 0.499715298414230350f,
  0.518403589725494380f, 0.499661177396774290f, 0.519936442375183110f,
  0.499602377414703370f, 0.521469116210937500f, 0.499538868665695190f,
  0.523001611232757570f, 0.499470651149749760f,
  0.524533808231353760f, 0.499397724866867070f, 0.526065826416015630f,
  0.499320119619369510f, 0.527597606182098390f, 0.499237775802612300f,
  0.529129147529602050f, 0.499150782823562620f,
  0.530660390853881840f, 0.499059051275253300f, 0.532191336154937740f,
  0.498962640762329100f, 0.533721983432769780f, 0.498861521482467650f,
  0.535252273082733150f, 0.498755723237991330f,
  0.536782264709472660f, 0.498645216226577760f, 0.538311958312988280f,
  0.498530030250549320f, 0.539841234683990480f, 0.498410135507583620f,
  0.541370153427124020f, 0.498285561800003050f,
  0.542898654937744140f, 0.498156309127807620f, 0.544426798820495610f,
  0.498022347688674930f, 0.545954465866088870f, 0.497883707284927370f,
  0.547481775283813480f, 0.497740387916564940f,
  0.549008548259735110f, 0.497592359781265260f, 0.550534904003143310f,
  0.497439652681350710f, 0.552060842514038090f, 0.497282296419143680f,
  0.553586184978485110f, 0.497120231389999390f,
  0.555111110210418700f, 0.496953487396240230f, 0.556635499000549320f,
  0.496782064437866210f, 0.558159291744232180f, 0.496605962514877320f,
  0.559682607650756840f, 0.496425211429595950f,
  0.561205327510833740f, 0.496239781379699710f, 0.562727510929107670f,
  0.496049642562866210f, 0.564249038696289060f, 0.495854884386062620f,
  0.565770030021667480f, 0.495655417442321780f,
  0.567290365695953370f, 0.495451331138610840f, 0.568810045719146730f,
  0.495242536067962650f, 0.570329129695892330f, 0.495029091835021970f,
  0.571847498416900630f, 0.494810998439788820f,
  0.573365211486816410f, 0.494588255882263180f, 0.574882268905639650f,
  0.494360834360122680f, 0.576398611068725590f, 0.494128793478012080f,
  0.577914178371429440f, 0.493892073631286620f,
  0.579429090023040770f, 0.493650704622268680f, 0.580943167209625240f,
  0.493404686450958250f, 0.582456588745117190f, 0.493154048919677730f,
  0.583969175815582280f, 0.492898762226104740f,
  0.585480928421020510f, 0.492638826370239260f, 0.586991965770721440f,
  0.492374241352081300f, 0.588502109050750730f, 0.492105036973953250f,
  0.590011477470397950f, 0.491831213235855100f,
  0.591519951820373540f, 0.491552740335464480f, 0.593027591705322270f,
  0.491269648075103760f, 0.594534337520599370f, 0.490981936454772950f,
  0.596040189266204830f, 0.490689605474472050f,
  0.597545146942138670f, 0.490392625331878660f, 0.599049210548400880f,
  0.490091055631637570f, 0.600552320480346680f, 0.489784896373748780f,
  0.602054476737976070f, 0.489474087953567500f,
  0.603555679321289060f, 0.489158689975738530f, 0.605055928230285640f,
  0.488838672637939450f, 0.606555163860321040f, 0.488514065742492680f,
  0.608053386211395260f, 0.488184869289398190f,
  0.609550595283508300f, 0.487851053476333620f, 0.611046791076660160f,
  0.487512677907943730f, 0.612541973590850830f, 0.487169682979583740f,
  0.614036023616790770f, 0.486822128295898440f,
  0.615529060363769530f, 0.486469984054565430f, 0.617020964622497560f,
  0.486113250255584720f, 0.618511795997619630f, 0.485751956701278690f,
  0.620001494884490970f, 0.485386073589324950f,
  0.621490061283111570f, 0.485015630722045900f, 0.622977554798126220f,
  0.484640628099441530f, 0.624463796615600590f, 0.484261035919189450f,
  0.625948905944824220f, 0.483876913785934450f,
  0.627432823181152340f, 0.483488231897354130f, 0.628915548324584960f,
  0.483094990253448490f, 0.630397081375122070f, 0.482697218656539920f,
  0.631877362728118900f, 0.482294887304306030f,
  0.633356392383575440f, 0.481888025999069210f, 0.634834170341491700f,
  0.481476634740829470f, 0.636310696601867680f, 0.481060713529586790f,
  0.637785911560058590f, 0.480640232563018800f,
  0.639259815216064450f, 0.480215251445770260f, 0.640732467174530030f,
  0.479785770177841190f, 0.642203748226165770f, 0.479351729154586790f,
  0.643673717975616460f, 0.478913217782974240f,
  0.645142316818237300f, 0.478470176458358760f, 0.646609604358673100f,
  0.478022634983062740f, 0.648075461387634280f, 0.477570593357086180f,
  0.649539887905120850f, 0.477114051580429080f,
  0.651003003120422360f, 0.476653009653091430f, 0.652464628219604490f,
  0.476187497377395630f, 0.653924822807312010f, 0.475717514753341670f,
  0.655383586883544920f, 0.475243031978607180f,
  0.656840860843658450f, 0.474764078855514530f, 0.658296704292297360f,
  0.474280685186386110f, 0.659750998020172120f, 0.473792791366577150f,
  0.661203861236572270f, 0.473300457000732420f,
  0.662655174732208250f, 0.472803652286529540f, 0.664104938507080080f,
  0.472302407026290890f, 0.665553152561187740f, 0.471796721220016480f,
  0.666999816894531250f, 0.471286594867706300f,
  0.668444931507110600f, 0.470772027969360350f, 0.669888436794281010f,
  0.470253020524978640f, 0.671330332756042480f, 0.469729602336883540f,
  0.672770678997039790f, 0.469201773405075070f,
  0.674209356307983400f, 0.468669503927230830f, 0.675646364688873290f,
  0.468132823705673220f, 0.677081763744354250f, 0.467591762542724610f,
  0.678515493869781490f, 0.467046260833740230f,
  0.679947495460510250f, 0.466496407985687260f, 0.681377887725830080f,
  0.465942144393920900f, 0.682806491851806640f, 0.465383470058441160f,
  0.684233427047729490f, 0.464820444583892820f,
  0.685658574104309080f, 0.464253038167953490f, 0.687082052230834960f,
  0.463681250810623170f, 0.688503682613372800f, 0.463105112314224240f,
  0.689923584461212160f, 0.462524622678756710f,
  0.691341698169708250f, 0.461939752101898190f, 0.692758023738861080f,
  0.461350560188293460f, 0.694172501564025880f, 0.460757017135620120f,
  0.695585191249847410f, 0.460159152746200560f,
  0.696996033191680910f, 0.459556937217712400f, 0.698404967784881590f,
  0.458950400352478030f, 0.699812114238739010f, 0.458339542150497440f,
  0.701217353343963620f, 0.457724362611770630f,
  0.702620685100555420f, 0.457104891538620000f, 0.704022109508514400f,
  0.456481099128723140f, 0.705421566963195800f, 0.455853015184402470f,
  0.706819176673889160f, 0.455220639705657960f,
  0.708214759826660160f, 0.454584002494812010f, 0.709608435630798340f,
  0.453943043947219850f, 0.711000144481658940f, 0.453297853469848630f,
  0.712389826774597170f, 0.452648371458053590f,
  0.713777542114257810f, 0.451994657516479490f, 0.715163230895996090f,
  0.451336652040481570f, 0.716546893119812010f, 0.450674414634704590f,
  0.717928528785705570f, 0.450007945299148560f,
  0.719308137893676760f, 0.449337244033813480f, 0.720685660839080810f,
  0.448662281036376950f, 0.722061097621917720f, 0.447983115911483760f,
  0.723434448242187500f, 0.447299748659133910f,
  0.724805653095245360f, 0.446612149477005000f, 0.726174771785736080f,
  0.445920348167419430f, 0.727541804313659670f, 0.445224374532699580f,
  0.728906631469726560f, 0.444524168968200680f,
  0.730269372463226320f, 0.443819820880889890f, 0.731629908084869380f,
  0.443111270666122440f, 0.732988238334655760f, 0.442398548126220700f,
  0.734344422817230220f, 0.441681683063507080f,
  0.735698342323303220f, 0.440960645675659180f, 0.737050116062164310f,
  0.440235435962677000f, 0.738399624824523930f, 0.439506113529205320f,
  0.739746868610382080f, 0.438772648572921750f,
  0.741091907024383540f, 0.438035041093826290f, 0.742434620857238770f,
  0.437293320894241330f, 0.743775069713592530f, 0.436547487974166870f,
  0.745113253593444820f, 0.435797542333602910f,
  0.746449112892150880f, 0.435043483972549440f, 0.747782647609710690f,
  0.434285342693328860f, 0.749113857746124270f, 0.433523118495941160f,
  0.750442683696746830f, 0.432756811380386350f,
  0.751769185066223140f, 0.431986421346664430f, 0.753093302249908450f,
  0.431211978197097780f, 0.754415094852447510f, 0.430433481931686400f,
  0.755734443664550780f, 0.429650902748107910f,
  0.757051348686218260f, 0.428864300251007080f, 0.758365929126739500f,
  0.428073674440383910f, 0.759678006172180180f, 0.427278995513916020f,
  0.760987639427185060f, 0.426480293273925780f,
  0.762294828891754150f, 0.425677597522735600f, 0.763599574565887450f,
  0.424870878458023070f, 0.764901816844940190f, 0.424060165882110600f,
  0.766201555728912350f, 0.423245459794998170f,
  0.767498791217803960f, 0.422426789999008180f, 0.768793523311614990f,
  0.421604126691818240f, 0.770085752010345460f, 0.420777499675750730f,
  0.771375417709350590f, 0.419946908950805660f,
  0.772662520408630370f, 0.419112354516983030f, 0.773947000503540040f,
  0.418273866176605220f, 0.775228977203369140f, 0.417431443929672240f,
  0.776508331298828130f, 0.416585087776184080f,
  0.777785122394561770f, 0.415734797716140750f, 0.779059290885925290f,
  0.414880603551864620f, 0.780330777168273930f, 0.414022535085678100f,
  0.781599700450897220f, 0.413160532712936400f,
  0.782865881919860840f, 0.412294656038284300f, 0.784129500389099120f,
  0.411424905061721800f, 0.785390377044677730f, 0.410551249980926510f,
  0.786648571491241460f, 0.409673750400543210f,
  0.787904083728790280f, 0.408792406320571900f, 0.789156913757324220f,
  0.407907217741012570f, 0.790407001972198490f, 0.407018154859542850f,
  0.791654348373413090f, 0.406125307083129880f,
  0.792898952960968020f, 0.405228585004806520f, 0.794140756130218510f,
  0.404328078031539920f, 0.795379877090454100f, 0.403423786163330080f,
  0.796616137027740480f, 0.402515679597854610f,
  0.797849655151367190f, 0.401603758335113530f, 0.799080371856689450f,
  0.400688081979751590f, 0.800308227539062500f, 0.399768620729446410f,
  0.801533281803131100f, 0.398845434188842770f,
  0.802755534648895260f, 0.397918462753295900f, 0.803974866867065430f,
  0.396987736225128170f, 0.805191397666931150f, 0.396053284406661990f,
  0.806405067443847660f, 0.395115107297897340f,
  0.807615816593170170f, 0.394173204898834230f, 0.808823645114898680f,
  0.393227607011795040f, 0.810028612613677980f, 0.392278283834457400f,
  0.811230659484863280f, 0.391325294971466060f,
  0.812429726123809810f, 0.390368610620498660f, 0.813625931739807130f,
  0.389408260583877560f, 0.814819097518920900f, 0.388444244861602780f,
  0.816009342670440670f, 0.387476563453674320f,
  0.817196667194366460f, 0.386505216360092160f, 0.818380951881408690f,
  0.385530263185501100f, 0.819562196731567380f, 0.384551674127578740f,
  0.820740520954132080f, 0.383569449186325070f,
  0.821915745735168460f, 0.382583618164062500f, 0.823087990283966060f,
  0.381594210863113400f, 0.824257194995880130f, 0.380601197481155400f,
  0.825423359870910640f, 0.379604607820510860f,
  0.826586425304412840f, 0.378604412078857420f, 0.827746450901031490f,
  0.377600699663162230f, 0.828903317451477050f, 0.376593410968780520f,
  0.830057144165039060f, 0.375582575798034670f,
  0.831207871437072750f, 0.374568194150924680f, 0.832355499267578130f,
  0.373550295829772950f, 0.833499968051910400f, 0.372528880834579470f,
  0.834641277790069580f, 0.371503978967666630f,
  0.835779488086700440f, 0.370475560426712040f, 0.836914479732513430f,
  0.369443655014038090f, 0.838046371936798100f, 0.368408292531967160f,
  0.839175045490264890f, 0.367369443178176880f,
  0.840300500392913820f, 0.366327136754989620f, 0.841422796249389650f,
  0.365281373262405400f, 0.842541813850402830f, 0.364232182502746580f,
  0.843657672405242920f, 0.363179564476013180f,
  0.844770252704620360f, 0.362123548984527590f, 0.845879614353179930f,
  0.361064106225967410f, 0.846985757350921630f, 0.360001266002655030f,
  0.848088562488555910f, 0.358935028314590450f,
  0.849188148975372310f, 0.357865422964096070f, 0.850284397602081300f,
  0.356792420148849490f, 0.851377367973327640f, 0.355716109275817870f,
  0.852467060089111330f, 0.354636400938034060f,
  0.853553414344787600f, 0.353553384542465210f, 0.854636430740356450f,
  0.352467030286788940f, 0.855716109275817870f, 0.351377367973327640f,
  0.856792449951171880f, 0.350284397602081300f,
  0.857865393161773680f, 0.349188119173049930f, 0.858934998512268070f,
  0.348088562488555910f, 0.860001266002655030f, 0.346985727548599240f,
  0.861064076423645020f, 0.345879614353179930f,
  0.862123548984527590f, 0.344770282506942750f, 0.863179564476013180f,
  0.343657672405242920f, 0.864232182502746580f, 0.342541843652725220f,
  0.865281403064727780f, 0.341422766447067260f,
  0.866327106952667240f, 0.340300500392913820f, 0.867369413375854490f,
  0.339175015687942500f, 0.868408262729644780f, 0.338046342134475710f,
  0.869443655014038090f, 0.336914509534835820f,
  0.870475590229034420f, 0.335779488086700440f, 0.871503949165344240f,
  0.334641307592391970f, 0.872528910636901860f, 0.333499968051910400f,
  0.873550295829772950f, 0.332355499267578130f,
  0.874568223953247070f, 0.331207901239395140f, 0.875582575798034670f,
  0.330057173967361450f, 0.876593410968780520f, 0.328903347253799440f,
  0.877600669860839840f, 0.327746421098709110f,
  0.878604412078857420f, 0.326586425304412840f, 0.879604578018188480f,
  0.325423330068588260f, 0.880601167678833010f, 0.324257194995880130f,
  0.881594181060791020f, 0.323088020086288450f,
  0.882583618164062500f, 0.321915775537490840f, 0.883569478988647460f,
  0.320740520954132080f, 0.884551644325256350f, 0.319562226533889770f,
  0.885530233383178710f, 0.318380922079086300f,
  0.886505246162414550f, 0.317196637392044070f, 0.887476563453674320f,
  0.316009372472763060f, 0.888444244861602780f, 0.314819127321243290f,
  0.889408230781555180f, 0.313625901937484740f,
  0.890368640422821040f, 0.312429755926132200f, 0.891325294971466060f,
  0.311230629682540890f, 0.892278313636779790f, 0.310028612613677980f,
  0.893227577209472660f, 0.308823645114898680f,
  0.894173204898834230f, 0.307615786790847780f, 0.895115137100219730f,
  0.306405037641525270f, 0.896053314208984380f, 0.305191397666931150f,
  0.896987736225128170f, 0.303974896669387820f,
  0.897918462753295900f, 0.302755534648895260f, 0.898845434188842770f,
  0.301533311605453490f, 0.899768650531768800f, 0.300308227539062500f,
  0.900688111782073970f, 0.299080342054367070f,
  0.901603758335113530f, 0.297849655151367190f, 0.902515649795532230f,
  0.296616137027740480f, 0.903423786163330080f, 0.295379847288131710f,
  0.904328107833862300f, 0.294140785932540890f,
  0.905228614807128910f, 0.292898923158645630f, 0.906125307083129880f,
  0.291654318571090700f, 0.907018184661865230f, 0.290406972169876100f,
  0.907907187938690190f, 0.289156883955001830f,
  0.908792436122894290f, 0.287904083728790280f, 0.909673750400543210f,
  0.286648571491241460f, 0.910551249980926510f, 0.285390377044677730f,
  0.911424875259399410f, 0.284129470586776730f,
  0.912294626235961910f, 0.282865911722183230f, 0.913160502910614010f,
  0.281599670648574830f, 0.914022505283355710f, 0.280330777168273930f,
  0.914880633354187010f, 0.279059261083602910f,
  0.915734827518463130f, 0.277785122394561770f, 0.916585087776184080f,
  0.276508361101150510f, 0.917431414127349850f, 0.275228977203369140f,
  0.918273866176605220f, 0.273947030305862430f,
  0.919112324714660640f, 0.272662490606307980f, 0.919946908950805660f,
  0.271375387907028200f, 0.920777499675750730f, 0.270085722208023070f,
  0.921604096889495850f, 0.268793523311614990f,
  0.922426760196685790f, 0.267498821020126340f, 0.923245489597320560f,
  0.266201555728912350f, 0.924060165882110600f, 0.264901816844940190f,
  0.924870908260345460f, 0.263599574565887450f,
  0.925677597522735600f, 0.262294828891754150f, 0.926480293273925780f,
  0.260987639427185060f, 0.927278995513916020f, 0.259678006172180180f,
  0.928073644638061520f, 0.258365899324417110f,
  0.928864300251007080f, 0.257051378488540650f, 0.929650902748107910f,
  0.255734413862228390f, 0.930433452129364010f, 0.254415065050125120f,
  0.931211948394775390f, 0.253093332052230830f,
  0.931986451148986820f, 0.251769185066223140f, 0.932756841182708740f,
  0.250442683696746830f, 0.933523118495941160f, 0.249113827943801880f,
  0.934285342693328860f, 0.247782632708549500f,
  0.935043513774871830f, 0.246449097990989690f, 0.935797572135925290f,
  0.245113238692283630f, 0.936547517776489260f, 0.243775084614753720f,
  0.937293350696563720f, 0.242434620857238770f,
  0.938035070896148680f, 0.241091892123222350f, 0.938772618770599370f,
  0.239746883511543270f, 0.939506113529205320f, 0.238399609923362730f,
  0.940235435962677000f, 0.237050101161003110f,
  0.940960645675659180f, 0.235698372125625610f, 0.941681683063507080f,
  0.234344407916069030f, 0.942398548126220700f, 0.232988253235816960f,
  0.943111240863800050f, 0.231629893183708190f,
  0.943819820880889890f, 0.230269357562065120f, 0.944524168968200680f,
  0.228906646370887760f, 0.945224344730377200f, 0.227541789412498470f,
  0.945920348167419430f, 0.226174786686897280f,
  0.946612179279327390f, 0.224805667996406560f, 0.947299718856811520f,
  0.223434418439865110f, 0.947983145713806150f, 0.222061067819595340f,
  0.948662281036376950f, 0.220685631036758420f,
  0.949337244033813480f, 0.219308122992515560f, 0.950007975101470950f,
  0.217928543686866760f, 0.950674414634704590f, 0.216546908020973210f,
  0.951336681842803960f, 0.215163245797157290f,
  0.951994657516479490f, 0.213777542114257810f, 0.952648401260375980f,
  0.212389841675758360f, 0.953297853469848630f, 0.211000129580497740f,
  0.953943073749542240f, 0.209608450531959530f,
  0.954584002494812010f, 0.208214774727821350f, 0.955220639705657960f,
  0.206819161772727970f, 0.955853044986724850f, 0.205421581864356990f,
  0.956481099128723140f, 0.204022079706192020f,
  0.957104861736297610f, 0.202620655298233030f, 0.957724332809448240f,
  0.201217323541641240f, 0.958339512348175050f, 0.199812099337577820f,
  0.958950400352478030f, 0.198404997587203980f,
  0.959556937217712400f, 0.196996018290519710f, 0.960159122943878170f,
  0.195585191249847410f, 0.960757017135620120f, 0.194172516465187070f,
  0.961350560188293460f, 0.192758023738861080f,
  0.961939752101898190f, 0.191341713070869450f, 0.962524592876434330f,
  0.189923599362373350f, 0.963105142116546630f, 0.188503712415695190f,
  0.963681280612945560f, 0.187082037329673770f,
  0.964253067970275880f, 0.185658603906631470f, 0.964820444583892820f,
  0.184233412146568300f, 0.965383470058441160f, 0.182806491851806640f,
  0.965942144393920900f, 0.181377857923507690f,
  0.966496407985687260f, 0.179947525262832640f, 0.967046260833740230f,
  0.178515478968620300f, 0.967591762542724610f, 0.177081763744354250f,
  0.968132853507995610f, 0.175646379590034480f,
  0.968669533729553220f, 0.174209341406822200f, 0.969201743602752690f,
  0.172770664095878600f, 0.969729602336883540f, 0.171330362558364870f,
  0.970253050327301030f, 0.169888436794281010f,
  0.970772027969360350f, 0.168444931507110600f, 0.971286594867706300f,
  0.166999831795692440f, 0.971796751022338870f, 0.165553152561187740f,
  0.972302436828613280f, 0.164104923605918880f,
  0.972803652286529540f, 0.162655144929885860f, 0.973300457000732420f,
  0.161203846335411070f, 0.973792791366577150f, 0.159751012921333310f,
  0.974280655384063720f, 0.158296689391136170f,
  0.974764108657836910f, 0.156840875744819640f, 0.975243031978607180f,
  0.155383571982383730f, 0.975717484951019290f, 0.153924822807312010f,
  0.976187527179718020f, 0.152464613318443300f,
  0.976653039455413820f, 0.151002973318099980f, 0.977114021778106690f,
  0.149539917707443240f, 0.977570593357086180f, 0.148075446486473080f,
  0.978022634983062740f, 0.146609574556350710f,
  0.978470146656036380f, 0.145142331719398500f, 0.978913187980651860f,
  0.143673732876777650f, 0.979351758956909180f, 0.142203763127326970f,
  0.979785740375518800f, 0.140732467174530030f,
  0.980215251445770260f, 0.139259845018386840f, 0.980640232563018800f,
  0.137785911560058590f, 0.981060683727264400f, 0.136310681700706480f,
  0.981476604938507080f, 0.134834155440330510f,
  0.981888055801391600f, 0.133356377482414250f, 0.982294917106628420f,
  0.131877332925796510f, 0.982697248458862300f, 0.130397051572799680f,
  0.983094990253448490f, 0.128915548324584960f,
  0.983488261699676510f, 0.127432823181152340f, 0.983876943588256840f,
  0.125948905944824220f, 0.984261035919189450f, 0.124463804066181180f,
  0.984640598297119140f, 0.122977524995803830f,
  0.985015630722045900f, 0.121490091085433960f, 0.985386073589324950f,
  0.120001509785652160f, 0.985751926898956300f, 0.118511803448200230f,
  0.986113250255584720f, 0.117020979523658750f,
  0.986469984054565430f, 0.115529052913188930f, 0.986822128295898440f,
  0.114036038517951970f, 0.987169682979583740f, 0.112541958689689640f,
  0.987512648105621340f, 0.111046813428401950f,
  0.987851083278656010f, 0.109550617635250090f, 0.988184869289398190f,
  0.108053401112556460f, 0.988514065742492680f, 0.106555156409740450f,
  0.988838672637939450f, 0.105055920779705050f,
  0.989158689975738530f, 0.103555686771869660f, 0.989474058151245120f,
  0.102054484188556670f, 0.989784896373748780f, 0.100552320480346680f,
  0.990091085433959960f, 0.099049203097820282f,
  0.990392625331878660f, 0.097545161843299866f, 0.990689575672149660f,
  0.096040196716785431f, 0.990981936454772950f, 0.094534330070018768f,
  0.991269648075103760f, 0.093027576804161072f,
  0.991552770137786870f, 0.091519944369792938f, 0.991831183433532710f,
  0.090011447668075562f, 0.992105066776275630f, 0.088502109050750732f,
  0.992374241352081300f, 0.086991935968399048f,
  0.992638826370239260f, 0.085480943322181702f, 0.992898762226104740f,
  0.083969146013259888f, 0.993154048919677730f, 0.082456558942794800f,
  0.993404686450958250f, 0.080943197011947632f,
  0.993650734424591060f, 0.079429075121879578f, 0.993892073631286620f,
  0.077914200723171234f, 0.994128763675689700f, 0.076398596167564392f,
  0.994360864162445070f, 0.074882268905639648f,
  0.994588255882263180f, 0.073365233838558197f, 0.994810998439788820f,
  0.071847513318061829f, 0.995029091835021970f, 0.070329122245311737f,
  0.995242536067962650f, 0.068810060620307922f,
  0.995451331138610840f, 0.067290350794792175f, 0.995655417442321780f,
  0.065770015120506287f, 0.995854854583740230f, 0.064249053597450256f,
  0.996049642562866210f, 0.062727488577365875f,
  0.996239781379699710f, 0.061205338686704636f, 0.996425211429595950f,
  0.059682607650756836f, 0.996605992317199710f, 0.058159314095973969f,
  0.996782064437866210f, 0.056635476648807526f,
  0.996953487396240230f, 0.055111102759838104f, 0.997120201587677000f,
  0.053586211055517197f, 0.997282266616821290f, 0.052060816437005997f,
  0.997439682483673100f, 0.050534930080175400f,
  0.997592389583587650f, 0.049008570611476898f, 0.997740387916564940f,
  0.047481749206781387f, 0.997883677482604980f, 0.045954477041959763f,
  0.998022377490997310f, 0.044426776468753815f,
  0.998156309127807620f, 0.042898654937744141f, 0.998285591602325440f,
  0.041370131075382233f, 0.998410165309906010f, 0.039841219782829285f,
  0.998530030250549320f, 0.038311932235956192f,
  0.998645246028900150f, 0.036782283335924149f, 0.998755753040313720f,
  0.035252287983894348f, 0.998861551284790040f, 0.033721961081027985f,
  0.998962640762329100f, 0.032191313803195953f,
  0.999059081077575680f, 0.030660368502140045f, 0.999150753021240230f,
  0.029129132628440857f, 0.999237775802612300f, 0.027597622945904732f,
  0.999320089817047120f, 0.026065852493047714f,
  0.999397754669189450f, 0.024533838033676147f, 0.999470651149749760f,
  0.023001590743660927f, 0.999538838863372800f, 0.021469129249453545f,
  0.999602377414703370f, 0.019936462864279747f,
  0.999661207199096680f, 0.018403612077236176f, 0.999715328216552730f,
  0.016870586201548576f, 0.999764680862426760f, 0.015337402001023293f,
  0.999809384346008300f, 0.013804072514176369f,
  0.999849438667297360f, 0.012270614504814148f, 0.999884724617004390f,
  0.010737040080130100f, 0.999915301799774170f, 0.009203365072607994f,
  0.999941170215606690f, 0.007669602986425161f,
  0.999962329864501950f, 0.006135769188404083f, 0.999978840351104740f,
  0.004601877182722092f, 0.999990582466125490f, 0.003067942336201668f,
  0.999997675418853760f, 0.001533978385850787f,
  1.000000000000000000f, 0.000000000000023345f, 0.999997675418853760f,
  -0.001533978385850787f, 0.999990582466125490f, -0.003067942336201668f,
  0.999978840351104740f, -0.004601877182722092f,
  0.999962329864501950f, -0.006135769188404083f, 0.999941170215606690f,
  -0.007669602986425161f, 0.999915301799774170f, -0.009203365072607994f,
  0.999884724617004390f, -0.010737040080130100f,
  0.999849438667297360f, -0.012270614504814148f, 0.999809384346008300f,
  -0.013804072514176369f, 0.999764680862426760f, -0.015337402001023293f,
  0.999715328216552730f, -0.016870586201548576f,
  0.999661207199096680f, -0.018403612077236176f, 0.999602377414703370f,
  -0.019936462864279747f, 0.999538838863372800f, -0.021469129249453545f,
  0.999470651149749760f, -0.023001590743660927f,
  0.999397754669189450f, -0.024533838033676147f, 0.999320089817047120f,
  -0.026065852493047714f, 0.999237775802612300f, -0.027597622945904732f,
  0.999150753021240230f, -0.029129132628440857f,
  0.999059081077575680f, -0.030660368502140045f, 0.998962640762329100f,
  -0.032191313803195953f, 0.998861551284790040f, -0.033721961081027985f,
  0.998755753040313720f, -0.035252287983894348f,
  0.998645246028900150f, -0.036782283335924149f, 0.998530030250549320f,
  -0.038311932235956192f, 0.998410165309906010f, -0.039841219782829285f,
  0.998285591602325440f, -0.041370131075382233f,
  0.998156309127807620f, -0.042898654937744141f, 0.998022377490997310f,
  -0.044426776468753815f, 0.997883677482604980f, -0.045954477041959763f,
  0.997740387916564940f, -0.047481749206781387f,
  0.997592389583587650f, -0.049008570611476898f, 0.997439682483673100f,
  -0.050534930080175400f, 0.997282266616821290f, -0.052060816437005997f,
  0.997120201587677000f, -0.053586211055517197f,
  0.996953487396240230f, -0.055111102759838104f, 0.996782064437866210f,
  -0.056635476648807526f, 0.996605992317199710f, -0.058159314095973969f,
  0.996425211429595950f, -0.059682607650756836f,
  0.996239781379699710f, -0.061205338686704636f, 0.996049642562866210f,
  -0.062727488577365875f, 0.995854854583740230f, -0.064249053597450256f,
  0.995655417442321780f, -0.065770015120506287f,
  0.995451331138610840f, -0.067290350794792175f, 0.995242536067962650f,
  -0.068810060620307922f, 0.995029091835021970f, -0.070329122245311737f,
  0.994810998439788820f, -0.071847513318061829f,
  0.994588255882263180f, -0.073365233838558197f, 0.994360864162445070f,
  -0.074882268905639648f, 0.994128763675689700f, -0.076398596167564392f,
  0.993892073631286620f, -0.077914200723171234f,
  0.993650734424591060f, -0.079429075121879578f, 0.993404686450958250f,
  -0.080943197011947632f, 0.993154048919677730f, -0.082456558942794800f,
  0.992898762226104740f, -0.083969146013259888f,
  0.992638826370239260f, -0.085480943322181702f, 0.992374241352081300f,
  -0.086991935968399048f, 0.992105066776275630f, -0.088502109050750732f,
  0.991831183433532710f, -0.090011447668075562f,
  0.991552770137786870f, -0.091519944369792938f, 0.991269648075103760f,
  -0.093027576804161072f, 0.990981936454772950f, -0.094534330070018768f,
  0.990689575672149660f, -0.096040196716785431f,
  0.990392625331878660f, -0.097545161843299866f, 0.990091085433959960f,
  -0.099049203097820282f, 0.989784896373748780f, -0.100552320480346680f,
  0.989474058151245120f, -0.102054484188556670f,
  0.989158689975738530f, -0.103555686771869660f, 0.988838672637939450f,
  -0.105055920779705050f, 0.988514065742492680f, -0.106555156409740450f,
  0.988184869289398190f, -0.108053401112556460f,
  0.987851083278656010f, -0.109550617635250090f, 0.987512648105621340f,
  -0.111046813428401950f, 0.987169682979583740f, -0.112541958689689640f,
  0.986822128295898440f, -0.114036038517951970f,
  0.986469984054565430f, -0.115529052913188930f, 0.986113250255584720f,
  -0.117020979523658750f, 0.985751926898956300f, -0.118511803448200230f,
  0.985386073589324950f, -0.120001509785652160f,
  0.985015630722045900f, -0.121490091085433960f, 0.984640598297119140f,
  -0.122977524995803830f, 0.984261035919189450f, -0.124463804066181180f,
  0.983876943588256840f, -0.125948905944824220f,
  0.983488261699676510f, -0.127432823181152340f, 0.983094990253448490f,
  -0.128915548324584960f, 0.982697248458862300f, -0.130397051572799680f,
  0.982294917106628420f, -0.131877332925796510f,
  0.981888055801391600f, -0.133356377482414250f, 0.981476604938507080f,
  -0.134834155440330510f, 0.981060683727264400f, -0.136310681700706480f,
  0.980640232563018800f, -0.137785911560058590f,
  0.980215251445770260f, -0.139259845018386840f, 0.979785740375518800f,
  -0.140732467174530030f, 0.979351758956909180f, -0.142203763127326970f,
  0.978913187980651860f, -0.143673732876777650f,
  0.978470146656036380f, -0.145142331719398500f, 0.978022634983062740f,
  -0.146609574556350710f, 0.977570593357086180f, -0.148075446486473080f,
  0.977114021778106690f, -0.149539917707443240f,
  0.976653039455413820f, -0.151002973318099980f, 0.976187527179718020f,
  -0.152464613318443300f, 0.975717484951019290f, -0.153924822807312010f,
  0.975243031978607180f, -0.155383571982383730f,
  0.974764108657836910f, -0.156840875744819640f, 0.974280655384063720f,
  -0.158296689391136170f, 0.973792791366577150f, -0.159751012921333310f,
  0.973300457000732420f, -0.161203846335411070f,
  0.972803652286529540f, -0.162655144929885860f, 0.972302436828613280f,
  -0.164104923605918880f, 0.971796751022338870f, -0.165553152561187740f,
  0.971286594867706300f, -0.166999831795692440f,
  0.970772027969360350f, -0.168444931507110600f, 0.970253050327301030f,
  -0.169888436794281010f, 0.969729602336883540f, -0.171330362558364870f,
  0.969201743602752690f, -0.172770664095878600f,
  0.968669533729553220f, -0.174209341406822200f, 0.968132853507995610f,
  -0.175646379590034480f, 0.967591762542724610f, -0.177081763744354250f,
  0.967046260833740230f, -0.178515478968620300f,
  0.966496407985687260f, -0.179947525262832640f, 0.965942144393920900f,
  -0.181377857923507690f, 0.965383470058441160f, -0.182806491851806640f,
  0.964820444583892820f, -0.184233412146568300f,
  0.964253067970275880f, -0.185658603906631470f, 0.963681280612945560f,
  -0.187082037329673770f, 0.963105142116546630f, -0.188503712415695190f,
  0.962524592876434330f, -0.189923599362373350f,
  0.961939752101898190f, -0.191341713070869450f, 0.961350560188293460f,
  -0.192758023738861080f, 0.960757017135620120f, -0.194172516465187070f,
  0.960159122943878170f, -0.195585191249847410f,
  0.959556937217712400f, -0.196996018290519710f, 0.958950400352478030f,
  -0.198404997587203980f, 0.958339512348175050f, -0.199812099337577820f,
  0.957724332809448240f, -0.201217323541641240f,
  0.957104861736297610f, -0.202620655298233030f, 0.956481099128723140f,
  -0.204022079706192020f, 0.955853044986724850f, -0.205421581864356990f,
  0.955220639705657960f, -0.206819161772727970f,
  0.954584002494812010f, -0.208214774727821350f, 0.953943073749542240f,
  -0.209608450531959530f, 0.953297853469848630f, -0.211000129580497740f,
  0.952648401260375980f, -0.212389841675758360f,
  0.951994657516479490f, -0.213777542114257810f, 0.951336681842803960f,
  -0.215163245797157290f, 0.950674414634704590f, -0.216546908020973210f,
  0.950007975101470950f, -0.217928543686866760f,
  0.949337244033813480f, -0.219308122992515560f, 0.948662281036376950f,
  -0.220685631036758420f, 0.947983145713806150f, -0.222061067819595340f,
  0.947299718856811520f, -0.223434418439865110f,
  0.946612179279327390f, -0.224805667996406560f, 0.945920348167419430f,
  -0.226174786686897280f, 0.945224344730377200f, -0.227541789412498470f,
  0.944524168968200680f, -0.228906646370887760f,
  0.943819820880889890f, -0.230269357562065120f, 0.943111240863800050f,
  -0.231629893183708190f, 0.942398548126220700f, -0.232988253235816960f,
  0.941681683063507080f, -0.234344407916069030f,
  0.940960645675659180f, -0.235698372125625610f, 0.940235435962677000f,
  -0.237050101161003110f, 0.939506113529205320f, -0.238399609923362730f,
  0.938772618770599370f, -0.239746883511543270f,
  0.938035070896148680f, -0.241091892123222350f, 0.937293350696563720f,
  -0.242434620857238770f, 0.936547517776489260f, -0.243775084614753720f,
  0.935797572135925290f, -0.245113238692283630f,
  0.935043513774871830f, -0.246449097990989690f, 0.934285342693328860f,
  -0.247782632708549500f, 0.933523118495941160f, -0.249113827943801880f,
  0.932756841182708740f, -0.250442683696746830f,
  0.931986451148986820f, -0.251769185066223140f, 0.931211948394775390f,
  -0.253093332052230830f, 0.930433452129364010f, -0.254415065050125120f,
  0.929650902748107910f, -0.255734413862228390f,
  0.928864300251007080f, -0.257051378488540650f, 0.928073644638061520f,
  -0.258365899324417110f, 0.927278995513916020f, -0.259678006172180180f,
  0.926480293273925780f, -0.260987639427185060f,
  0.925677597522735600f, -0.262294828891754150f, 0.924870908260345460f,
  -0.263599574565887450f, 0.924060165882110600f, -0.264901816844940190f,
  0.923245489597320560f, -0.266201555728912350f,
  0.922426760196685790f, -0.267498821020126340f, 0.921604096889495850f,
  -0.268793523311614990f, 0.920777499675750730f, -0.270085722208023070f,
  0.919946908950805660f, -0.271375387907028200f,
  0.919112324714660640f, -0.272662490606307980f, 0.918273866176605220f,
  -0.273947030305862430f, 0.917431414127349850f, -0.275228977203369140f,
  0.916585087776184080f, -0.276508361101150510f,
  0.915734827518463130f, -0.277785122394561770f, 0.914880633354187010f,
  -0.279059261083602910f, 0.914022505283355710f, -0.280330777168273930f,
  0.913160502910614010f, -0.281599670648574830f,
  0.912294626235961910f, -0.282865911722183230f, 0.911424875259399410f,
  -0.284129470586776730f, 0.910551249980926510f, -0.285390377044677730f,
  0.909673750400543210f, -0.286648571491241460f,
  0.908792436122894290f, -0.287904083728790280f, 0.907907187938690190f,
  -0.289156883955001830f, 0.907018184661865230f, -0.290406972169876100f,
  0.906125307083129880f, -0.291654318571090700f,
  0.905228614807128910f, -0.292898923158645630f, 0.904328107833862300f,
  -0.294140785932540890f, 0.903423786163330080f, -0.295379847288131710f,
  0.902515649795532230f, -0.296616137027740480f,
  0.901603758335113530f, -0.297849655151367190f, 0.900688111782073970f,
  -0.299080342054367070f, 0.899768650531768800f, -0.300308227539062500f,
  0.898845434188842770f, -0.301533311605453490f,
  0.897918462753295900f, -0.302755534648895260f, 0.896987736225128170f,
  -0.303974896669387820f, 0.896053314208984380f, -0.305191397666931150f,
  0.895115137100219730f, -0.306405037641525270f,
  0.894173204898834230f, -0.307615786790847780f, 0.893227577209472660f,
  -0.308823645114898680f, 0.892278313636779790f, -0.310028612613677980f,
  0.891325294971466060f, -0.311230629682540890f,
  0.890368640422821040f, -0.312429755926132200f, 0.889408230781555180f,
  -0.313625901937484740f, 0.888444244861602780f, -0.314819127321243290f,
  0.887476563453674320f, -0.316009372472763060f,
  0.886505246162414550f, -0.317196637392044070f, 0.885530233383178710f,
  -0.318380922079086300f, 0.884551644325256350f, -0.319562226533889770f,
  0.883569478988647460f, -0.320740520954132080f,
  0.882583618164062500f, -0.321915775537490840f, 0.881594181060791020f,
  -0.323088020086288450f, 0.880601167678833010f, -0.324257194995880130f,
  0.879604578018188480f, -0.325423330068588260f,
  0.878604412078857420f, -0.326586425304412840f, 0.877600669860839840f,
  -0.327746421098709110f, 0.876593410968780520f, -0.328903347253799440f,
  0.875582575798034670f, -0.330057173967361450f,
  0.874568223953247070f, -0.331207901239395140f, 0.873550295829772950f,
  -0.332355499267578130f, 0.872528910636901860f, -0.333499968051910400f,
  0.871503949165344240f, -0.334641307592391970f,
  0.870475590229034420f, -0.335779488086700440f, 0.869443655014038090f,
  -0.336914509534835820f, 0.868408262729644780f, -0.338046342134475710f,
  0.867369413375854490f, -0.339175015687942500f,
  0.866327106952667240f, -0.340300500392913820f, 0.865281403064727780f,
  -0.341422766447067260f, 0.864232182502746580f, -0.342541843652725220f,
  0.863179564476013180f, -0.343657672405242920f,
  0.862123548984527590f, -0.344770282506942750f, 0.861064076423645020f,
  -0.345879614353179930f, 0.860001266002655030f, -0.346985727548599240f,
  0.858934998512268070f, -0.348088562488555910f,
  0.857865393161773680f, -0.349188119173049930f, 0.856792449951171880f,
  -0.350284397602081300f, 0.855716109275817870f, -0.351377367973327640f,
  0.854636430740356450f, -0.352467030286788940f,
  0.853553414344787600f, -0.353553384542465210f, 0.852467060089111330f,
  -0.354636400938034060f, 0.851377367973327640f, -0.355716109275817870f,
  0.850284397602081300f, -0.356792420148849490f,
  0.849188148975372310f, -0.357865422964096070f, 0.848088562488555910f,
  -0.358935028314590450f, 0.846985757350921630f, -0.360001266002655030f,
  0.845879614353179930f, -0.361064106225967410f,
  0.844770252704620360f, -0.362123548984527590f, 0.843657672405242920f,
  -0.363179564476013180f, 0.842541813850402830f, -0.364232182502746580f,
  0.841422796249389650f, -0.365281373262405400f,
  0.840300500392913820f, -0.366327136754989620f, 0.839175045490264890f,
  -0.367369443178176880f, 0.838046371936798100f, -0.368408292531967160f,
  0.836914479732513430f, -0.369443655014038090f,
  0.835779488086700440f, -0.370475560426712040f, 0.834641277790069580f,
  -0.371503978967666630f, 0.833499968051910400f, -0.372528880834579470f,
  0.832355499267578130f, -0.373550295829772950f,
  0.831207871437072750f, -0.374568194150924680f, 0.830057144165039060f,
  -0.375582575798034670f, 0.828903317451477050f, -0.376593410968780520f,
  0.827746450901031490f, -0.377600699663162230f,
  0.826586425304412840f, -0.378604412078857420f, 0.825423359870910640f,
  -0.379604607820510860f, 0.824257194995880130f, -0.380601197481155400f,
  0.823087990283966060f, -0.381594210863113400f,
  0.821915745735168460f, -0.382583618164062500f, 0.820740520954132080f,
  -0.383569449186325070f, 0.819562196731567380f, -0.384551674127578740f,
  0.818380951881408690f, -0.385530263185501100f,
  0.817196667194366460f, -0.386505216360092160f, 0.816009342670440670f,
  -0.387476563453674320f, 0.814819097518920900f, -0.388444244861602780f,
  0.813625931739807130f, -0.389408260583877560f,
  0.812429726123809810f, -0.390368610620498660f, 0.811230659484863280f,
  -0.391325294971466060f, 0.810028612613677980f, -0.392278283834457400f,
  0.808823645114898680f, -0.393227607011795040f,
  0.807615816593170170f, -0.394173204898834230f, 0.806405067443847660f,
  -0.395115107297897340f, 0.805191397666931150f, -0.396053284406661990f,
  0.803974866867065430f, -0.396987736225128170f,
  0.802755534648895260f, -0.397918462753295900f, 0.801533281803131100f,
  -0.398845434188842770f, 0.800308227539062500f, -0.399768620729446410f,
  0.799080371856689450f, -0.400688081979751590f,
  0.797849655151367190f, -0.401603758335113530f, 0.796616137027740480f,
  -0.402515679597854610f, 0.795379877090454100f, -0.403423786163330080f,
  0.794140756130218510f, -0.404328078031539920f,
  0.792898952960968020f, -0.405228585004806520f, 0.791654348373413090f,
  -0.406125307083129880f, 0.790407001972198490f, -0.407018154859542850f,
  0.789156913757324220f, -0.407907217741012570f,
  0.787904083728790280f, -0.408792406320571900f, 0.786648571491241460f,
  -0.409673750400543210f, 0.785390377044677730f, -0.410551249980926510f,
  0.784129500389099120f, -0.411424905061721800f,
  0.782865881919860840f, -0.412294656038284300f, 0.781599700450897220f,
  -0.413160532712936400f, 0.780330777168273930f, -0.414022535085678100f,
  0.779059290885925290f, -0.414880603551864620f,
  0.777785122394561770f, -0.415734797716140750f, 0.776508331298828130f,
  -0.416585087776184080f, 0.775228977203369140f, -0.417431443929672240f,
  0.773947000503540040f, -0.418273866176605220f,
  0.772662520408630370f, -0.419112354516983030f, 0.771375417709350590f,
  -0.419946908950805660f, 0.770085752010345460f, -0.420777499675750730f,
  0.768793523311614990f, -0.421604126691818240f,
  0.767498791217803960f, -0.422426789999008180f, 0.766201555728912350f,
  -0.423245459794998170f, 0.764901816844940190f, -0.424060165882110600f,
  0.763599574565887450f, -0.424870878458023070f,
  0.762294828891754150f, -0.425677597522735600f, 0.760987639427185060f,
  -0.426480293273925780f, 0.759678006172180180f, -0.427278995513916020f,
  0.758365929126739500f, -0.428073674440383910f,
  0.757051348686218260f, -0.428864300251007080f, 0.755734443664550780f,
  -0.429650902748107910f, 0.754415094852447510f, -0.430433481931686400f,
  0.753093302249908450f, -0.431211978197097780f,
  0.751769185066223140f, -0.431986421346664430f, 0.750442683696746830f,
  -0.432756811380386350f, 0.749113857746124270f, -0.433523118495941160f,
  0.747782647609710690f, -0.434285342693328860f,
  0.746449112892150880f, -0.435043483972549440f, 0.745113253593444820f,
  -0.435797542333602910f, 0.743775069713592530f, -0.436547487974166870f,
  0.742434620857238770f, -0.437293320894241330f,
  0.741091907024383540f, -0.438035041093826290f, 0.739746868610382080f,
  -0.438772648572921750f, 0.738399624824523930f, -0.439506113529205320f,
  0.737050116062164310f, -0.440235435962677000f,
  0.735698342323303220f, -0.440960645675659180f, 0.734344422817230220f,
  -0.441681683063507080f, 0.732988238334655760f, -0.442398548126220700f,
  0.731629908084869380f, -0.443111270666122440f,
  0.730269372463226320f, -0.443819820880889890f, 0.728906631469726560f,
  -0.444524168968200680f, 0.727541804313659670f, -0.445224374532699580f,
  0.726174771785736080f, -0.445920348167419430f,
  0.724805653095245360f, -0.446612149477005000f, 0.723434448242187500f,
  -0.447299748659133910f, 0.722061097621917720f, -0.447983115911483760f,
  0.720685660839080810f, -0.448662281036376950f,
  0.719308137893676760f, -0.449337244033813480f, 0.717928528785705570f,
  -0.450007945299148560f, 0.716546893119812010f, -0.450674414634704590f,
  0.715163230895996090f, -0.451336652040481570f,
  0.713777542114257810f, -0.451994657516479490f, 0.712389826774597170f,
  -0.452648371458053590f, 0.711000144481658940f, -0.453297853469848630f,
  0.709608435630798340f, -0.453943043947219850f,
  0.708214759826660160f, -0.454584002494812010f, 0.706819176673889160f,
  -0.455220639705657960f, 0.705421566963195800f, -0.455853015184402470f,
  0.704022109508514400f, -0.456481099128723140f,
  0.702620685100555420f, -0.457104891538620000f, 0.701217353343963620f,
  -0.457724362611770630f, 0.699812114238739010f, -0.458339542150497440f,
  0.698404967784881590f, -0.458950400352478030f,
  0.696996033191680910f, -0.459556937217712400f, 0.695585191249847410f,
  -0.460159152746200560f, 0.694172501564025880f, -0.460757017135620120f,
  0.692758023738861080f, -0.461350560188293460f,
  0.691341698169708250f, -0.461939752101898190f, 0.689923584461212160f,
  -0.462524622678756710f, 0.688503682613372800f, -0.463105112314224240f,
  0.687082052230834960f, -0.463681250810623170f,
  0.685658574104309080f, -0.464253038167953490f, 0.684233427047729490f,
  -0.464820444583892820f, 0.682806491851806640f, -0.465383470058441160f,
  0.681377887725830080f, -0.465942144393920900f,
  0.679947495460510250f, -0.466496407985687260f, 0.678515493869781490f,
  -0.467046260833740230f, 0.677081763744354250f, -0.467591762542724610f,
  0.675646364688873290f, -0.468132823705673220f,
  0.674209356307983400f, -0.468669503927230830f, 0.672770678997039790f,
  -0.469201773405075070f, 0.671330332756042480f, -0.469729602336883540f,
  0.669888436794281010f, -0.470253020524978640f,
  0.668444931507110600f, -0.470772027969360350f, 0.666999816894531250f,
  -0.471286594867706300f, 0.665553152561187740f, -0.471796721220016480f,
  0.664104938507080080f, -0.472302407026290890f,
  0.662655174732208250f, -0.472803652286529540f, 0.661203861236572270f,
  -0.473300457000732420f, 0.659750998020172120f, -0.473792791366577150f,
  0.658296704292297360f, -0.474280685186386110f,
  0.656840860843658450f, -0.474764078855514530f, 0.655383586883544920f,
  -0.475243031978607180f, 0.653924822807312010f, -0.475717514753341670f,
  0.652464628219604490f, -0.476187497377395630f,
  0.651003003120422360f, -0.476653009653091430f, 0.649539887905120850f,
  -0.477114051580429080f, 0.648075461387634280f, -0.477570593357086180f,
  0.646609604358673100f, -0.478022634983062740f,
  0.645142316818237300f, -0.478470176458358760f, 0.643673717975616460f,
  -0.478913217782974240f, 0.642203748226165770f, -0.479351729154586790f,
  0.640732467174530030f, -0.479785770177841190f,
  0.639259815216064450f, -0.480215251445770260f, 0.637785911560058590f,
  -0.480640232563018800f, 0.636310696601867680f, -0.481060713529586790f,
  0.634834170341491700f, -0.481476634740829470f,
  0.633356392383575440f, -0.481888025999069210f, 0.631877362728118900f,
  -0.482294887304306030f, 0.630397081375122070f, -0.482697218656539920f,
  0.628915548324584960f, -0.483094990253448490f,
  0.627432823181152340f, -0.483488231897354130f, 0.625948905944824220f,
  -0.483876913785934450f, 0.624463796615600590f, -0.484261035919189450f,
  0.622977554798126220f, -0.484640628099441530f,
  0.621490061283111570f, -0.485015630722045900f, 0.620001494884490970f,
  -0.485386073589324950f, 0.618511795997619630f, -0.485751956701278690f,
  0.617020964622497560f, -0.486113250255584720f,
  0.615529060363769530f, -0.486469984054565430f, 0.614036023616790770f,
  -0.486822128295898440f, 0.612541973590850830f, -0.487169682979583740f,
  0.611046791076660160f, -0.487512677907943730f,
  0.609550595283508300f, -0.487851053476333620f, 0.608053386211395260f,
  -0.488184869289398190f, 0.606555163860321040f, -0.488514065742492680f,
  0.605055928230285640f, -0.488838672637939450f,
  0.603555679321289060f, -0.489158689975738530f, 0.602054476737976070f,
  -0.489474087953567500f, 0.600552320480346680f, -0.489784896373748780f,
  0.599049210548400880f, -0.490091055631637570f,
  0.597545146942138670f, -0.490392625331878660f, 0.596040189266204830f,
  -0.490689605474472050f, 0.594534337520599370f, -0.490981936454772950f,
  0.593027591705322270f, -0.491269648075103760f,
  0.591519951820373540f, -0.491552740335464480f, 0.590011477470397950f,
  -0.491831213235855100f, 0.588502109050750730f, -0.492105036973953250f,
  0.586991965770721440f, -0.492374241352081300f,
  0.585480928421020510f, -0.492638826370239260f, 0.583969175815582280f,
  -0.492898762226104740f, 0.582456588745117190f, -0.493154048919677730f,
  0.580943167209625240f, -0.493404686450958250f,
  0.579429090023040770f, -0.493650704622268680f, 0.577914178371429440f,
  -0.493892073631286620f, 0.576398611068725590f, -0.494128793478012080f,
  0.574882268905639650f, -0.494360834360122680f,
  0.573365211486816410f, -0.494588255882263180f, 0.571847498416900630f,
  -0.494810998439788820f, 0.570329129695892330f, -0.495029091835021970f,
  0.568810045719146730f, -0.495242536067962650f,
  0.567290365695953370f, -0.495451331138610840f, 0.565770030021667480f,
  -0.495655417442321780f, 0.564249038696289060f, -0.495854884386062620f,
  0.562727510929107670f, -0.496049642562866210f,
  0.561205327510833740f, -0.496239781379699710f, 0.559682607650756840f,
  -0.496425211429595950f, 0.558159291744232180f, -0.496605962514877320f,
  0.556635499000549320f, -0.496782064437866210f,
  0.555111110210418700f, -0.496953487396240230f, 0.553586184978485110f,
  -0.497120231389999390f, 0.552060842514038090f, -0.497282296419143680f,
  0.550534904003143310f, -0.497439652681350710f,
  0.549008548259735110f, -0.497592359781265260f, 0.547481775283813480f,
  -0.497740387916564940f, 0.545954465866088870f, -0.497883707284927370f,
  0.544426798820495610f, -0.498022347688674930f,
  0.542898654937744140f, -0.498156309127807620f, 0.541370153427124020f,
  -0.498285561800003050f, 0.539841234683990480f, -0.498410135507583620f,
  0.538311958312988280f, -0.498530030250549320f,
  0.536782264709472660f, -0.498645216226577760f, 0.535252273082733150f,
  -0.498755723237991330f, 0.533721983432769780f, -0.498861521482467650f,
  0.532191336154937740f, -0.498962640762329100f,
  0.530660390853881840f, -0.499059051275253300f, 0.529129147529602050f,
  -0.499150782823562620f, 0.527597606182098390f, -0.499237775802612300f,
  0.526065826416015630f, -0.499320119619369510f,
  0.524533808231353760f, -0.499397724866867070f, 0.523001611232757570f,
  -0.499470651149749760f, 0.521469116210937500f, -0.499538868665695190f,
  0.519936442375183110f, -0.499602377414703370f,
  0.518403589725494380f, -0.499661177396774290f, 0.516870558261871340f,
  -0.499715298414230350f, 0.515337407588958740f, -0.499764710664749150f,
  0.513804078102111820f, -0.499809414148330690f,
  0.512270629405975340f, -0.499849408864974980f, 0.510737061500549320f,
  -0.499884694814682010f, 0.509203374385833740f, -0.499915301799774170f,
  0.507669627666473390f, -0.499941170215606690f,
  0.506135761737823490f, -0.499962359666824340f, 0.504601895809173580f,
  -0.499978810548782350f, 0.503067970275878910f, -0.499990582466125490f,
  0.501533985137939450f, -0.499997645616531370f
};



/**   
* @brief  Initialization function for the floating-point RFFT/RIFFT.  
* @param[in,out] *S             points to an instance of the floating-point RFFT/RIFFT structure.  
* @param[in,out] *S_CFFT        points to an instance of the floating-point CFFT/CIFFT structure.  
* @param[in]     fftLenReal     length of the FFT.  
* @param[in]     ifftFlagR      flag that selects forward (ifftFlagR=0) or inverse (ifftFlagR=1) transform.  
* @param[in]     bitReverseFlag flag that enables (bitReverseFlag=1) or disables (bitReverseFlag=0) bit reversal of output.  
* @return		The function returns ARM_MATH_SUCCESS if initialization is successful or ARM_MATH_ARGUMENT_ERROR if <code>fftLenReal</code> is not a supported value.  
*   
* \par Description:  
* \par  
* The parameter <code>fftLenReal</code>	Specifies length of RFFT/RIFFT Process. Supported FFT Lengths are 128, 512, 2048.   
* \par   
* The parameter <code>ifftFlagR</code> controls whether a forward or inverse transform is computed.   
* Set(=1) ifftFlagR to calculate RIFFT, otherwise RFFT is calculated.   
* \par   
* The parameter <code>bitReverseFlag</code> controls whether output is in normal order or bit reversed order.   
* Set(=1) bitReverseFlag for output to be in normal order otherwise output is in bit reversed order.  
* \par   
* This function also initializes Twiddle factor table.    
*/

arm_status arm_rfft_init_f32(
  arm_rfft_instance_f32 * S,
  arm_cfft_radix4_instance_f32 * S_CFFT,
  uint32_t fftLenReal,
  uint32_t ifftFlagR,
  uint32_t bitReverseFlag)
{

  /*  Initialise the default arm status */
  arm_status status = ARM_MATH_SUCCESS;

  /*  Initialize the Real FFT length */
  S->fftLenReal = (uint16_t) fftLenReal;

  /*  Initialize the Complex FFT length */
  S->fftLenBy2 = (uint16_t) fftLenReal / 2u;

  /*  Initialize the Twiddle coefficientA pointer */
  S->pTwiddleAReal = (float32_t *) realCoefA;

  /*  Initialize the Twiddle coefficientB pointer */
  S->pTwiddleBReal = (float32_t *) realCoefB;

  /*  Initialize the Flag for selection of RFFT or RIFFT */
  S->ifftFlagR = (uint8_t) ifftFlagR;

  /*  Initialize the Flag for calculation Bit reversal or not */
  S->bitReverseFlagR = (uint8_t) bitReverseFlag;

  /*  Initializations of structure parameters depending on the FFT length */
  switch (S->fftLenReal)
  {
    /* Init table modifier value */
  case 2048u:
    S->twidCoefRModifier = 1u;
    break;
  case 512u:
    S->twidCoefRModifier = 4u;
    break;
  case 128u:
    S->twidCoefRModifier = 16u;
    break;
  default:
    /*  Reporting argument error if rfftSize is not valid value */
    status = ARM_MATH_ARGUMENT_ERROR;
    break;
  }

  /* Init Complex FFT Instance */
  S->pCfft = S_CFFT;

  if(S->ifftFlagR)
  {
    /* Initializes the CIFFT Module for fftLenreal/2 length */
    arm_cfft_radix4_init_f32(S->pCfft, S->fftLenBy2, 1u, 0u);
  }
  else
  {
    /* Initializes the CFFT Module for fftLenreal/2 length */
    arm_cfft_radix4_init_f32(S->pCfft, S->fftLenBy2, 0u, 0u);
  }

  /* return the status of RFFT Init function */
  return (status);

}

  /**   
   * @} end of RFFT_RIFFT group   
   */
