/**************************************************************************************************
 * Files module
 *
 * Author: Razvan Madalin MATEI <matei.rm94@gmail.com
 * Date last modified: April 2015
 *************************************************************************************************/

#ifdef FILES

#define FUSE_USE_VERSION 30
#include <fuse.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <pthread.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <strophe.h>

#include "../winternals/winternals.h" /* logs and errs */
#include "../wxmpp/wxmpp.h"           /* stanzas       */
#include "../base64/base64.h"         /* decoe         */
#include "../libds/ds.h"              /* hashmap       */

#include "files.h"

extern xmpp_ctx_t *global_ctx;   /* Context    */
extern xmpp_conn_t *global_conn; /* Connection */

extern const char *owner; /* owner from init.c */
extern const char *mount_file; /* mount file */

extern bool is_owner_online; /* connection checker from wxmpp_handlers.c */

pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER; /* mutex      */
pthread_cond_t  cond  = PTHREAD_COND_INITIALIZER;  /* condition  */

/* Condition signals */
bool signal_attr = false;
bool signal_list = false;
bool signal_read = false;
bool signal_fail = false;

/* Filetype */
typedef enum {
  DIR,          /* Directory    */
  REG           /* Regular file */
} filetype_t;

/* File attributes */
typedef struct {
  unsigned int size;
  filetype_t type;
  char is_valid;
} attr_t;

attr_t attributes = {0, DIR, -1};

/* List of files */
typedef struct elem_t {
  filetype_t type;
  char *filename;
  struct elem_t *next;
} elem_t;

/* root and last element of list of files */
elem_t *root = NULL;
elem_t *last = NULL;

char *read_data = NULL;

static void files_attr(xmpp_stanza_t *stanza);
static void files_list(xmpp_stanza_t *stanza);
static void files_read(xmpp_stanza_t *stanza);

static int wfuse_getattr(const char *path, struct stat *stbuf) {
  if (signal_fail == true) {
    /* Connection was lost. Now it's ok */
    signal_attr = false;
    signal_list = false;
    signal_read = false;
    signal_fail = false;
  }

  if (!is_owner_online) {
    return -ENOENT;
  }

  int rc; /* Return code */

  rc = pthread_mutex_lock(&mutex);
  wsyserr(rc != 0, "pthread_mutex_lock");

  /* Send attributes stanza */
  xmpp_stanza_t *message = xmpp_stanza_new(global_ctx); /* message with files */
  xmpp_stanza_set_name(message, "message");
  xmpp_stanza_set_attribute(message, "to", owner);

  xmpp_stanza_t *files = xmpp_stanza_new(global_ctx); /* files */
  xmpp_stanza_set_name(files, "files");
  xmpp_stanza_set_ns(files, WNS);
  xmpp_stanza_set_attribute(files, "action", "attributes");
  xmpp_stanza_set_attribute(files, "path", path);

  xmpp_stanza_add_child(message, files);
  xmpp_send(global_conn, message);
  xmpp_stanza_release(files);
  xmpp_stanza_release(message);

  /* Wait until attributes is set */
  while (signal_attr == false) {
    pthread_cond_wait(&cond, &mutex);
  }

  int res = 0;

  /* Do your job */
  if (signal_fail == true) {
    res = -ENOENT;

    signal_attr = false;
    signal_list = false;
    signal_read = false;
    signal_fail = false;
  } else {
    memset(stbuf, 0, sizeof(struct stat));
    if (strcmp(path, "/") == 0) {
      stbuf->st_mode = S_IFDIR | 0777;
      stbuf->st_nlink = 2;
    }

    else {
      if (attributes.is_valid == 1) {
        if (attributes.type == DIR) {
          stbuf->st_mode = S_IFDIR | 0777;
          stbuf->st_nlink = 2;
          stbuf->st_size = attributes.size;
        } else if (attributes.type == REG) {
          stbuf->st_mode = S_IFREG | 0444;
          stbuf->st_nlink = 1;
          stbuf->st_size = attributes.size;
        } else {
          werr("Unknown type");
          res = -ENOENT;
        }
      } else {
        res = -ENOENT;
      }
    }
  }
  /* Job done */

  signal_attr = false;
  rc = pthread_mutex_unlock(&mutex);
  wsyserr(rc != 0, "pthread_mutex_unlock");

  return res;
}

static int wfuse_readdir(const char *path, void *buf, fuse_fill_dir_t filler,
                         off_t offset, struct fuse_file_info *fi) {
  if (signal_fail == true) {
    /* Connection was lost. Now it's ok */
    signal_attr = false;
    signal_list = false;
    signal_read = false;
    signal_fail = false;
  }

  if (!is_owner_online) {
    return -ENOENT;
  }

  int rc; /* Return code */

  rc = pthread_mutex_lock(&mutex);
  wsyserr(rc != 0, "pthread_mutex_lock");

  xmpp_stanza_t *message = xmpp_stanza_new(global_ctx); /* message with done */
  xmpp_stanza_set_name(message, "message");
  xmpp_stanza_set_attribute(message, "to", owner);

  xmpp_stanza_t *files = xmpp_stanza_new(global_ctx); /* message with done */
  xmpp_stanza_set_name(files, "files");
  xmpp_stanza_set_ns(files, WNS);
  xmpp_stanza_set_attribute(files, "action", "list");
  xmpp_stanza_set_attribute(files, "path", path);

  xmpp_stanza_add_child(message, files);
  xmpp_send(global_conn, message);
  xmpp_stanza_release(files);
  xmpp_stanza_release(message);

  while (signal_list == false) {
    pthread_cond_wait(&cond, &mutex);
  }

  int res = 0;

  /* Do your job */
  if (signal_fail == true) {
    // res = -ENOENT;

    filler(buf, ".", NULL, 0);
    filler(buf, "..", NULL, 0);

    signal_attr = false;
    signal_list = false;
    signal_read = false;
    signal_fail = false;
  } else {

    (void) offset;
    (void) fi;

    filler(buf, ".", NULL, 0);
    filler(buf, "..", NULL, 0);

    elem_t *aux = root;
    while (aux != NULL) {
      filler(buf, aux->filename, NULL, 0);
      aux = aux->next;
    }

    /* Free list */
    aux = root;
    elem_t *aux2;
    while (aux != NULL) {
      free(aux->filename);
      aux2 = aux;
      aux = aux->next;
      free(aux2);
    }

    root = NULL;
  }
  /* Job done */

  signal_list = false;
  rc = pthread_mutex_unlock(&mutex);
  wsyserr(rc != 0, "pthread_mutex_unlock");

  return res;
}

static int wfuse_open(const char *path, struct fuse_file_info *fi) {
  wlog("wfuse_open path = %s\n", path);

  if (!is_owner_online) {
    werr("Owner not available");
    return -ENOENT;
  }

  if ((fi->flags & 3) != O_RDONLY)
    return -EACCES;

  return 0;
}

static int wfuse_read(const char *path, char *buf, size_t size, off_t offset,
                      struct fuse_file_info *fi) {
  if (signal_fail == true) {
    /* Connection was lost. Now it's ok */
    signal_attr = false;
    signal_list = false;
    signal_read = false;
    signal_fail = false;
  }

  if (!is_owner_online) {
    return -ENOENT;
  }

  int rc; /* Return code */

  rc = pthread_mutex_lock(&mutex);
  wsyserr(rc != 0, "pthread_mutex_lock");

  xmpp_stanza_t *message = xmpp_stanza_new(global_ctx); /* message with done */
  xmpp_stanza_set_name(message, "message");
  xmpp_stanza_set_attribute(message, "to", owner);

  xmpp_stanza_t *files = xmpp_stanza_new(global_ctx); /* message with done */
  xmpp_stanza_set_name(files, "files");
  xmpp_stanza_set_ns(files, WNS);
  xmpp_stanza_set_attribute(files, "action", "read");
  xmpp_stanza_set_attribute(files, "path", path);

  xmpp_stanza_add_child(message, files);
  xmpp_send(global_conn, message);
  xmpp_stanza_release(files);
  xmpp_stanza_release(message);

  /* Wait until attributes is set */
  while (signal_read == false) {
    pthread_cond_wait(&cond, &mutex);
  }

  int res = size;

  /* Do your job */
  if (signal_fail == true) {
    memset(buf, 0, size);

    signal_attr = false;
    signal_list = false;
    signal_read = false;
    signal_fail = false;
  } else {
    size_t len;
    (void) fi;
    len = strlen(read_data);

    if (offset < len) {
      if (offset + size > len) {
        size = len - offset;
      }
      memcpy(buf, read_data + offset, size);
    } else {
      size = 0;
    }
  }
  /* Job done */

  signal_read = false;

  rc = pthread_mutex_unlock(&mutex);
  wsyserr(rc != 0, "pthread_mutex_unlock");

  return res;
}

static struct fuse_operations wfuse_oper = {
  .getattr = wfuse_getattr,
  .readdir = wfuse_readdir,
  .open    = wfuse_open,
  .read    = wfuse_read
};

void *init_files_thread(void *a) {
  char *argv[] = {"dummy", "-s", "-f", (char *)mount_file};
  fuse_main(4, argv, &wfuse_oper, NULL);
  return NULL;
}

bool files_initialized = false;

void init_files() {
  if (!files_initialized) {
    pthread_t ift; /* Init files thread */
    int rc = pthread_create(&ift, NULL, init_files_thread, NULL);
    wsyserr(rc < 0, "pthread_create");

    files_initialized = true;
  }
}

void files(const char *from, const char *to, int error, xmpp_stanza_t *stanza,
           xmpp_conn_t *const conn, void *const userdata)
{
  wlog("files()");
  if (error == 0) {
    char *action_attr = xmpp_stanza_get_attribute(stanza, "action"); /* action attribute */
    if (action_attr == NULL) {
      werr("xmpp_stanza_get_attribute attribute = action");
    }
    werr2(action_attr == NULL, return, "There is no action attribute in file stanza");

    if (strcmp(action_attr, "attributes") == 0) {
      files_attr(stanza);
    } else if (strcmp(action_attr, "list") == 0) {
      files_list(stanza);
    } else if (strncasecmp(action_attr, "read", 4) == 0) {
      files_read(stanza);
    } else {
      werr("Unknown action: %s", action_attr);
    }
  } else {
    werr("error stanza %s %s", xmpp_stanza_get_attribute(stanza, "path"), xmpp_stanza_get_attribute(stanza, "action"));
  }

  wlog("Return from files()");
}

static void files_attr(xmpp_stanza_t *stanza) {
  int rc; /* Return code */

  rc = pthread_mutex_lock(&mutex);
  wsyserr(rc != 0, "pthread_mutex_lock");

  char *error_attr = xmpp_stanza_get_attribute(stanza, "error"); /* error attribute */
  werr2(error_attr == NULL, return, "There is no error attribute in files stanza");


  if (strcmp(error_attr, "0") != 0) {
    wlog("Error in attributes: %s", error_attr);
    attributes.is_valid = -1;
  } else {
    char *type_attr = xmpp_stanza_get_attribute(stanza, "type"); /* type attribute */
    werr2(type_attr == NULL, return, "There is no type attribute in files stanza");

    if (strcmp(type_attr, "directory") == 0) {
      attributes.is_valid = 1;
      attributes.type = DIR;
    } else if (strcmp(type_attr, "file") == 0) {
      attributes.is_valid = 1;
      attributes.type = REG;
    } else {
      werr("Unknown type: %s", type_attr);
      attributes.is_valid = -1;
    }

    char *size_attr = xmpp_stanza_get_attribute(stanza, "size"); /* size */
    if (size_attr == NULL) {
      // werr("xmpp_stanza_get_attribute attribute = size (%s)", xmpp_stanza_get_attribute(stanza, "path"));
      attributes.size = 0;
    } else {
      char *endptr; /* strtol endptr */
      long int size = strtol(size_attr, &endptr, 10);
      if (*endptr != '\0') {
        werr("strtol error: str = %s, val = %ld", size_attr, size);
        attributes.size = 0;
      }
      attributes.size = size;
    }
  }

  signal_attr = true;

  rc = pthread_cond_signal(&cond);
  wsyserr(rc != 0, "pthread_cond_signal");

  rc = pthread_mutex_unlock(&mutex);
  wsyserr(rc != 0, "pthread_mutex_unlock");
}

static void files_list(xmpp_stanza_t *stanza) {
  int rc; /* Return code */

  rc = pthread_mutex_lock(&mutex);
  wsyserr(rc != 0, "pthread_mutex_lock");

  char *error_attr = xmpp_stanza_get_attribute(stanza, "error"); /* error attribute */
  werr2(error_attr == NULL, return, "There is no error attribute in files stanza");

  if (strcmp(error_attr, "0") != 0) {
    wlog("Error in attributes: %s", error_attr);
  } else {
    char *path_attr = xmpp_stanza_get_attribute(stanza, "path"); /* path attribute */
    werr2(path_attr == NULL, return, "There is no path attribute in files stanza");

    xmpp_stanza_t *child = xmpp_stanza_get_children(stanza);
    while (child != NULL) {
      elem_t *elem = (elem_t *)malloc(sizeof(elem_t));
      wsyserr(elem == NULL, "malloc");

      elem->next = NULL;

      char *name = xmpp_stanza_get_name(child);
      werr2(name == NULL, return, "There is no name attribute in files stanza");

      if (strcmp(name, "directory") == 0) {
        elem->type = DIR;
      } else if (strcmp(name, "file") == 0) {
        elem->type = REG;
      } else {
        werr("Unknown name: %s", name);
      }

      char *filename_attr = xmpp_stanza_get_attribute(child, "filename");
      werr2(filename_attr == NULL, return, "There is no filename attribute in files stanza");

      elem->filename = strdup(filename_attr);
      wsyserr(elem->filename == NULL, "strdup");

      /* Add elem in list */
      if (root == NULL) {
        root = elem;
      } else {
        last->next = elem;
      }
      last = elem;

      child = xmpp_stanza_get_next(child);
    }
  }

  /* Data set */
  signal_list = true;

  rc = pthread_cond_signal(&cond);
  wsyserr(rc != 0, "pthread_cond_signal");

  rc = pthread_mutex_unlock(&mutex);
  wsyserr(rc != 0, "pthread_mutex_unlock");
}

static void files_read(xmpp_stanza_t *stanza) {
  int rc;

  rc = pthread_mutex_lock(&mutex);
  wsyserr(rc != 0, "pthread_mutex_lock");

  char *text = xmpp_stanza_get_text(stanza);
  if(text == NULL) {
    werr("xmpp_stanza_get_text returned NULL (%s, error %s)", xmpp_stanza_get_attribute(stanza, "path"), xmpp_stanza_get_attribute(stanza, "error"));
    read_data = strdup ("");
  }
  else
  {
    int dec_size = strlen(text) * 3 / 4 + 1;
    uint8_t *dec_text = (uint8_t *)calloc(dec_size, sizeof(uint8_t));
    rc = base64_decode(dec_text, text, dec_size);
    werr2(rc < 0, return, "Could not decode");

    if (read_data != NULL) {
      free(read_data);
    }
    read_data = strdup((char *)dec_text);
    wsyserr(read_data == NULL, "strdup");

    free(text);
    free(dec_text);
  }

  signal_read = true;
  rc = pthread_cond_signal(&cond);
  wsyserr(rc != 0, "pthread_cond_signal");

  rc = pthread_mutex_unlock(&mutex);
  wsyserr(rc != 0, "pthread_mutex_unlock");
}

#endif /* FILES */
