var five = require("../lib/johnny-five");
var board = new five.Board();
var scale = five.Fn.scale;

board.on("ready", function() {
  var cache = new Map();
  var index = 0;
  var sensors = {
    Light: function(data) {
      return (1023 - data) * 10 / data;
    },
    Rotary: function(data) {
      return scale(data, 0, 1023, 0, 100);
    },
    Temperature: function(data) {
      return Thermistor.f(data);
    }
  };

  Object.keys(sensors).forEach(function(key, i) {
    new five.Sensor("A" + i).on("data", function() {

      if (key === "Rotary") {
        console.log(this.value);
      }
      cache.set(i, {
        key: key,
        value: sensors[key](this.value)
      });

      if (i === index) {
        display();
      }
    });
  });

  /**
   * Device       => Grove Shield Pin Jack
   * -------------------------------------
   * Rotary       => A0
   * Temperature  => A2
   * Light        => A3
   * Button       => D3
   * LCD          => I2C
   *
   */

  var button = new five.Button(3);

  var lcd = new five.LCD({
    controller: "JHD1313M1"
  });

  /**
   * LCD Display Map:
   *
   *  Rotary:
   *    XXXX
   *  Sound:
   *    XXXX
   *  Temperature:
   *    XXXX
   *  Light:
   *    XXXX
   *
   * Change displayed data by pressing the button.
   *
   */

  function display() {
    var output = cache.get(index);
    var formatted = output.value.toFixed(2);

    // console.log(formatted);


    if (display.index !== index) {
      lcd.clear().cursor(0, 0).print(output.key);
    }

    if (display.formatted !== formatted) {
      lcd.cursor(1, 0).print(formatted);
    }

    display.formatted = formatted;
    display.index = index;
  }

  button.on("press", function() {
    index++;

    if (index === cache.size) {
      index = 0;
    }
  });
});


var Thermistor = (function() {
  var adcres, beta, kelvin, rb, ginf;

  adcres = 1023;
  // Beta parameter
  beta = 3975;
  // 0°C = 273.15 K
  kelvin = 273.15;
  // 10 kOhm (sensor resistance)
  rb = 10000;
  // Ginf = 1/Rinf
  ginf = 120.6685;

  return {
    c: function(raw) {
      var rthermistor, tempc;

      rthermistor = rb * (adcres / raw - 1);
      tempc = beta / (Math.log(rthermistor * ginf));

      return tempc - kelvin;
    },
    f: function(raw) {
      return (this.c(raw) * 9) / 5 + 32;
    }
  };
}());

// @markdown
// For this program, you'll need:
//
// ![Grove Base Shield v2](http://www.seeedstudio.com/depot/images/product/base%20shield%20V2_01.jpg)
//
// ![Grove - LCD RGB w/ Backlight](http://www.seeedstudio.com/wiki/images/0/03/Serial_LEC_RGB_Backlight_Lcd.jpg)
//
// ![Grove - Rotary Angle Module](http://www.seeedstudio.com/depot/images/product/rotary.jpg)
//
// ![Grove - Touch Module](http://www.seeedstudio.com/depot/images/P3202396_01.jpg)
//
// ![Grove - Button Module](http://www.seeedstudio.com/depot/images/product/bgpushb1.jpg)
//
// ![Grove - Sound Module](http://www.seeedstudio.com/depot/images/P2100722.jpg)
//
//
//
//
// @markdown


